<?php
// Load notes data from the JSON file
$notesJsonPath = 'data/notes.json';
$galleryJsonPath = 'data/gallery.json';
$phonesJsonPath = 'data/phones.json';

// Check if the notes file exists
if (file_exists($notesJsonPath)) {
    $notesJson = file_get_contents($notesJsonPath);
    $notes = json_decode($notesJson, true);
} else {
    // Default notes if the file doesn't exist
    $notes = [
        [
            'id' => 1,
            'title' => "January 5, 2024",
            'content' => "EMMA COLLINS' PERSONAL DIARY\n\nInvestigative notes and collected documents\n\nJanuary 5, 2024\nI finally returned to Ravenwood after years away. Everything seems familiar, yet different somehow. I feel the town is shrouded in a strange silence. I started reviewing old newspapers and local archives; some past events seem to have been deliberately forgotten. There are many buried secrets. I plan to start at the library, perhaps Helen Brooks can help me, despite her coldness since I returned. I remember her well, despite the distance, and I need to better understand her recent behavior. Is my return bothering someone?\n\n[Old local newspaper clipping]\n\"Santa Cruz Orphanage closed after mysterious incident - several children missing or deceased. Police do not fully clarify the case.\" (09/21/1997)",
            'date' => "01/05/2024",
            'preview' => "I finally returned to Ravenwood after years away. Everything seems familiar, yet different somehow..."
        ],
        [
            'id' => 2,
            'title' => "January 8, 2024",
            'content' => "January 8, 2024\nToday I discovered something devastating about my father. He ran over a couple, leaving two girls orphaned—Helen and her sister. I never imagined something like this about my own family. The accident never had serious legal consequences for him, something that now seems incomprehensible. I now understand the animosity Helen shows; she clearly never forgave it. Today I realized she almost blames me personally for it. How can I deal with this situation? I need to find a way to talk to her again.\n\n[Old local newspaper clipping]\n\"Fatal hit-and-run leaves Brooks couple dead in Ravenwood; driver Robert Collins involved, no formal charges.\" (06/04/1992)",
            'date' => "01/08/2024",
            'preview' => "Today I discovered something devastating about my father. He ran over a couple, leaving two girls orphaned—Helen and her sister..."
        ],
        [
            'id' => 3,
            'title' => "January 12, 2024",
            'content' => "January 12, 2024\nI spoke again with Daniel about the unauthorized sales of family properties. He seemed desperate. He finally confessed his gambling debts, and they appear to be more serious than I thought. My lawyer says it might be too late to recover the lost assets, but I feel obligated to try. Daniel needs urgent help; I can't imagine how far he would go to get rid of these problems. I received anonymous threats today related to his debts. This is becoming dangerous and concerning.\n\n[Bank document showing recent sales of Collins family property]",
            'date' => "01/12/2024",
            'preview' => "I spoke again with Daniel about the unauthorized sales of family properties. He seemed desperate..."
        ],
        [
            'id' => 4,
            'title' => "January 19, 2024",
            'content' => "January 19, 2024\nYale officially responded to my request about Sarah today. No records found. How could she maintain this lie for so long? This calls into question many things about her, perhaps even our relationship since childhood. I avoided confronting her for now, as I felt Sarah is too tense lately. Something serious must be happening behind this perfect facade she created.\n\n[Official negative response from Yale University about Sarah Mitchell]",
            'date' => "01/19/2024",
            'preview' => "Yale officially responded to my request about Sarah today. No records found. How could she maintain this lie..."
        ],
        [
            'id' => 5,
            'title' => "January 25, 2024",
            'content' => "January 25, 2024\nI discovered suspicious financial transfers made by Reverend Shaw. He seems to be diverting funds to international accounts. When mentioning this briefly in a casual conversation, I noticed his evident uneasiness. He definitely knows I'm investigating. What exactly is he hiding? It doesn't seem like something simple. I need to proceed carefully with this investigation, as the Reverend has much respect in town, and this could backfire on me if I'm not cautious.\n\n[Copy of suspicious financial statements from the local church]",
            'date' => "01/25/2024",
            'preview' => "I discovered suspicious financial transfers made by Reverend Shaw. He seems to be diverting funds..."
        ],
        [
            'id' => 6,
            'title' => "February 3, 2024",
            'content' => "February 3, 2024\nThe occultist collection in the library is truly impressive and disturbing. Helen showed deep uneasiness when I asked about some specific symbols. She clearly understands more about this than she appears to. Something very obscure may be happening in this town involving dangerous rituals or beliefs. Could this be connected to the old orphanage? Perhaps Helen seeks answers or revenge through these practices. I need to continue my research discreetly, avoiding raising even more suspicions.\n\n[Photocopy of occultist book page found in the municipal library]",
            'date' => "02/03/2024",
            'preview' => "The occultist collection in the library is truly impressive and disturbing. Helen showed deep uneasiness..."
        ],
        [
            'id' => 7,
            'title' => "February 10, 2024",
            'content' => "February 10, 2024\nI found old police reports suggesting irregularities in Sheriff Reed's actions. He seems to be involved in or at least tolerating illegal operations in town. Unexplained releases, confusing reports. Reed noticed my interest, clearly followed me yesterday after my visit to the station. I need to be extra careful when investigating this matter. Something serious might be behind this.\n\n[Photocopies of suspicious police reports signed by Sheriff Reed]",
            'date' => "02/10/2024",
            'preview' => "I found old police reports suggesting irregularities in Sheriff Reed's actions. He seems to be involved..."
        ],
        [
            'id' => 8,
            'title' => "February 20, 2024",
            'content' => "February 20, 2024\nThe annual donations made by someone with the initials M.T. (Mary Thompson) always occur on the exact date of the incident at the orphanage. Strange and disturbing. It seems like a recurring act of compensation for something that happened there, something someone doesn't want to forget or perhaps overcome. I still don't know exactly why, but this might be directly linked to someone powerful in town. I need to discover more about this urgently.\n\n[Financial record of recurring annual donations signed by M.T.]",
            'date' => "02/20/2024",
            'preview' => "The annual donations made by someone with the initials M.T. (Mary Thompson) always occur on the exact date of the incident at the orphanage..."
        ],
        [
            'id' => 9,
            'title' => "February 28, 2024",
            'content' => "February 28, 2024\nSarah is clearly avoiding contact with me. Maybe she realized I'm getting close to the truth about her false Yale story. I can't understand the reason for such a big lie. Is it just vanity or something deeper and more concerning? Whatever it is, I need to be careful when talking to her again, as she seems quite uncomfortable with my presence lately.",
            'date' => "02/28/2024",
            'preview' => "Sarah is clearly avoiding contact with me. Maybe she realized I'm getting close to the truth about her false story..."
        ],
        [
            'id' => 10,
            'title' => "March 5, 2024",
            'content' => "March 5, 2024\nI have a meeting scheduled tomorrow night at the old church. The source asked to keep everything in absolute secrecy. It might be my only chance to clarify everything once and for all. I believe I'm close to the truth, but I need to be extremely careful. Tomorrow could be decisive, in many different ways.",
            'date' => "03/05/2024",
            'preview' => "I have a meeting scheduled tomorrow night at the old church. The source asked to keep everything in absolute secrecy..."
        ]
    ];
    
    // Create directory if it doesn't exist
    if (!is_dir('data')) {
        mkdir('data', 0755, true);
    }
    
    // Save the default notes to the JSON file
    file_put_contents($notesJsonPath, json_encode($notes, JSON_PRETTY_PRINT));
}

// Check if the gallery file exists
if (file_exists($galleryJsonPath)) {
    $galleryJson = file_get_contents($galleryJsonPath);
    $gallery = json_decode($galleryJson, true);
} else {
    // Default gallery if the file doesn't exist
    $gallery = [
        [
            'id' => 1,
            'title' => 'Orphanage Document',
            'description' => 'Document found in municipal archives showing disturbing details about the incident at Santa Cruz Orphanage.',
            'image' => 'data/images/document1.jpg',
            'date' => '01/10/2024'
        ],
        [
            'id' => 2,
            'title' => 'Occult Symbols',
            'description' => 'Strange symbols found on the wall of the old church. Related to rituals mentioned in library books.',
            'image' => 'data/images/symbols.jpg',
            'date' => '02/04/2024'
        ],
        [
            'id' => 3,
            'title' => 'Bank Transfers',
            'description' => 'Receipt of suspicious value transfers sent by Reverend Shaw to an overseas account.',
            'image' => 'data/images/transfer.jpg',
            'date' => '01/27/2024'
        ],
        [
            'id' => 4,
            'title' => 'Accident Site',
            'description' => 'Old photo of the location where the accident that killed Helen\'s parents occurred. The case was quickly archived.',
            'image' => 'data/images/accident.jpg',
            'date' => '01/09/2024'
        ],
        [
            'id' => 5,
            'title' => 'Diary Page',
            'description' => 'Torn diary page found in the abandoned house near the orphanage. The writing appears to be from a child.',
            'image' => 'data/images/diary.jpg',
            'date' => '02/15/2024'
        ],
        [
            'id' => 6,
            'title' => 'Donation List',
            'description' => 'List of annual donations made by M.T. always on the same date as the orphanage incident.',
            'image' => 'data/images/donations.jpg',
            'date' => '02/21/2024'
        ]
    ];
    
    // Create directories if they don't exist
    if (!is_dir('data')) {
        mkdir('data', 0755, true);
    }
    if (!is_dir('data/images')) {
        mkdir('data/images', 0755, true);
    }
    
    // Save the default gallery to the JSON file
    file_put_contents($galleryJsonPath, json_encode($gallery, JSON_PRETTY_PRINT));
}

// Check if the calls file exists
if (file_exists($phonesJsonPath)) {
    $phonesJson = file_get_contents($phonesJsonPath);
    $phones = json_decode($phonesJson, true);
} else {
    // Default calls if the file doesn't exist
    $phones = [
        [
            'id' => 1,
            'name' => 'Daniel Collins',
            'number' => '(555) 123-4567',
            'date' => '01/12/2024',
            'time' => '14:45',
            'duration' => '2:31',
            'type' => 'outgoing',
            'audio' => 'data/audio/daniel_call.mp3',
            'description' => 'Tense conversation about family property sales and his gambling debts.'
        ],
        [
            'id' => 2,
            'name' => 'Unknown',
            'number' => 'Private Number',
            'date' => '01/20/2024',
            'time' => '22:17',
            'duration' => '1:45',
            'type' => 'incoming',
            'audio' => 'data/audio/unknown_call.mp3',
            'description' => 'Distorted voice making veiled threats about my investigation.'
        ],
        [
            'id' => 3,
            'name' => 'Sarah Mitchell',
            'number' => '(555) 987-6543',
            'date' => '02/28/2024',
            'time' => '16:05',
            'duration' => '3:26',
            'type' => 'outgoing',
            'audio' => 'data/audio/sarah_call.mp3',
            'description' => 'Attempt to schedule a meeting to talk about Yale. Sarah was visibly nervous.'
        ],
        [
            'id' => 4,
            'name' => 'Helen Brooks',
            'number' => '(555) 234-5678',
            'date' => '03/03/2024',
            'time' => '11:23',
            'duration' => '4:12',
            'type' => 'incoming',
            'audio' => 'data/audio/helen_call.mp3',
            'description' => 'Helen talking about rare books in the library and mentioning the orphanage in an enigmatic way.'
        ],
        [
            'id' => 5,
            'name' => 'Anonymous Source',
            'number' => 'Blocked Number',
            'date' => '03/05/2024',
            'time' => '09:14',
            'duration' => '1:03',
            'type' => 'incoming',
            'audio' => 'data/audio/anonymous_call.mp3',
            'description' => 'Scheduling a meeting at the old church to reveal "the whole truth".'
        ]
    ];
    
    // Create directories if they don't exist
    if (!is_dir('data')) {
        mkdir('data', 0755, true);
    }
    if (!is_dir('data/audio')) {
        mkdir('data/audio', 0755, true);
    }
    
    // Save the default calls to the JSON file
    file_put_contents($phonesJsonPath, json_encode($phones, JSON_PRETTY_PRINT));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Virtual iPhone 5s with Notes</title>
    <style>
        /* Reset and global variables */
        :root {
            --bg-color: #1c1c1e;
            --sidebar-bg: #2c2c2e;
            --text-color: #ffffff;
            --secondary-text: #8e8e93;
            --highlight-color: #ff9500;
            --border-color: #38383a;
            --note-bg: #1c1c1e;
            --iphone-black: #1d1d1d;
            --iphone-border: #616161;
            --iphone-silver: #e4e4e2;
            --message-sent-bg: #0b93f6;
            --message-received-bg: #3a3a3c;
            --email-bg: #121212;
            --email-header-bg: #1a1a1a;
            --email-text-color: #e0e0e0;
            --email-accent: #4dabf7;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;
        }
        
        body {
            background-color: #f0f0f0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            padding: 0;
            background: linear-gradient(135deg, #303030, #121212);
            overflow: hidden;
        }
        
        /* iPhone 7 Plus Device Styling */
        .iphone-container {
            position: relative;
            width: 375px;
            height: 667px;
            background-color: #1d1d1d;
            border-radius: 42px;
            box-shadow: 0 0 50px rgba(0, 0, 0, 0.5);
            border: 2px solid #222;
            overflow: hidden;
            transition: all 0.3s ease;
            margin: 0 auto;
            max-height: 90vh;
        }
        
        /* Mobile optimization */
        /* For really large devices (tablets) */
        @media (min-width: 768px) and (min-height: 1000px) {
            .iphone-container {
                width: 500px;
                height: 900px;
                border-radius: 55px;
            }
            
            .iphone-home-button {
                width: 70px;
                height: 70px;
            }
            
            .iphone-home-button::after {
                width: 50px;
                height: 50px;
            }
        }
        
        @media (max-width: 768px) {
            .iphone-container {
                width: 280px;
                height: 500px;
                border-radius: 20px;
                border: 4px solid var(--iphone-silver);
                margin: 10px auto;
                transform: scale(0.9);
                position: fixed;
                top: 50%;
                left: 50%;
                transform: translate(-50%, -50%);
            }
            
            body {
                padding: 0;
                height: 100vh;
                width: 100%;
                overflow: hidden;
                position: relative;
            }
            
            .iphone-top-bezel, .iphone-bottom-bezel {
                height: 40px;
            }
            
            .iphone-screen {
                top: 40px;
                height: calc(100% - 80px);
            }
        }
        
        /* iPhone 5s Top Bezel */
        .iphone-top-bezel {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 50px; /* Reduced from 60px */
            background-color: black;
            z-index: 5;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        .iphone-speaker {
            width: 40px;
            height: 6px;
            background-color: #333;
            border-radius: 3px;
            margin-bottom: 6px;
        }
        
        .iphone-camera {
            width: 8px;
            height: 8px;
            background-color: #333;
            border-radius: 50%;
            position: absolute;
            top: 22px;
            right: 100px;
        }
        
        /* iPhone 5s Bottom Bezel and Home Button */
        .iphone-bottom-bezel {
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 50px; /* Reduced from 60px */
            background-color: black;
            z-index: 5;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        .iphone-home-button {
            width: 36px; /* Reduced from 40px */
            height: 36px; /* Reduced from 40px */
            border: 2px solid #444;
            border-radius: 50%;
            background-color: white;
            cursor: pointer;
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 100;
        }
        
        .iphone-home-button:active {
            background-color: #ddd;
        }
        
        .iphone-home-button-square {
            width: 10px; /* Reduced from 12px */
            height: 10px; /* Reduced from 12px */
            border: 1px solid #999;
            border-radius: 2px;
        }
        
        /* iPhone Screen */
        .iphone-screen {
            position: absolute;
            top: 50px; /* Adjusted from 60px */
            left: 0;
            width: 100%;
            height: calc(100% - 100px); /* Adjusted from 120px */
            background-color: black;
            overflow: hidden;
        }
        
        /* Remove iPhone 5s side buttons */
        .power-button, .volume-up, .volume-down {
            display: none;
        }
        
        /* Add side buttons for iPhone 7 */
        .iphone7-power-button {
            position: absolute;
            right: -2px;
            top: 120px;
            width: 3px;
            height: 60px;
            background-color: #222;
            z-index: 6;
        }
        
        .iphone7-volume-up {
            position: absolute;
            left: -2px;
            top: 100px;
            width: 3px;
            height: 40px;
            background-color: #222;
            z-index: 6;
        }
        
        .iphone7-volume-down {
            position: absolute;
            left: -2px;
            top: 150px;
            width: 3px;
            height: 40px;
            background-color: #222;
            z-index: 6;
        }
        
        .iphone7-mute-switch {
            position: absolute;
            left: -2px;
            top: 60px;
            width: 3px;
            height: 20px;
            background-color: #222;
            z-index: 6;
        }
        
        /* Status Bar - IMPROVED */
        .status-bar {
            height: 20px;
            width: 100%;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0 5px;
            color: white;
            font-size: 12px;
            font-weight: 500;
            background-color: transparent;
            position: absolute;
            top: 0;
            left: 0;
            z-index: 10;
        }
        
        .status-carrier {
            flex: 1;
            text-align: left;
            padding-left: 5px;
        }
        
        .status-time {
            flex: 2;
            text-align: center;
        }
        
        .status-icons {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: flex-end;
            gap: 3px;
            padding-right: 5px;
        }
        
        /* Lock Screen - IMPROVED SLIDER */
        .lock-screen {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: url('https://images.pexels.com/photos/2559941/pexels-photo-2559941.jpeg');
            background-size: cover;
            background-position: center;
            display: flex;
            flex-direction: column;
            align-items: center;
            z-index: 20;
        }
        
        .lock-time {
            font-size: 60px; /* Reduced from 70px */
            font-weight: 200;
            color: white;
            margin-top: 70px; /* Reduced from 80px */
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.5);
        }
        
        .lock-date {
            font-size: 16px; /* Reduced from 18px */
            color: white;
            margin-top: 5px;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.5);
        }
        
        /* Improved slide to unlock - horizontal style */
        .slide-to-unlock {
            position: absolute;
            bottom: 70px; /* Reduced from 80px */
            width: 80%;
            height: 36px; /* Reduced from 40px */
            background-color: rgba(0, 0, 0, 0.4);
            border-radius: 18px; /* Adjusted to match height */
            display: flex;
            align-items: center;
            padding: 0 10px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.5);
        }
        
        .slide-arrow {
            width: 46px; /* Reduced from 50px */
            height: 28px; /* Reduced from 30px */
            background-color: rgba(255, 255, 255, 0.8);
            border-radius: 14px; /* Adjusted to match height */
            display: flex;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            user-select: none;
            position: absolute;
            left: 5px;
            box-shadow: 0 0 5px rgba(0, 0, 0, 0.3);
        }
        
        .slide-arrow::after {
            content: "→";
            font-size: 20px;
            color: #666;
        }
        
        .slide-text {
            width: 100%;
            text-align: center;
            color: rgba(255, 255, 255, 0.8);
            font-size: 16px; /* Reduced from 18px */
            font-weight: 300;
            letter-spacing: 1px;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.7);
        }
        
        .camera-icon {
            position: absolute;
            bottom: 70px; /* Reduced from 80px */
            right: 15px;
            width: 36px; /* Reduced from 40px */
            height: 36px; /* Reduced from 40px */
            background-color: rgba(0, 0, 0, 0.4);
            border-radius: 18px; /* Adjusted to match height */
            display: flex;
            justify-content: center;
            align-items: center;
            font-size: 18px; /* Reduced from 20px */
        }
        
        .passcode-screen {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            display: flex;
            flex-direction: column;
            align-items: center;
            z-index: 25;
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.3s ease;
        }
        
        .passcode-screen.active {
            opacity: 1;
            pointer-events: all;
        }
        
        .passcode-title {
            font-size: 18px; /* Reduced from 20px */
            color: white;
            margin-top: 50px; /* Reduced from 60px */
            margin-bottom: 20px;
        }
        
        .passcode-dots {
            display: flex;
            gap: 15px;
            margin-bottom: 28px; /* Reduced from 30px */
        }
        
        .passcode-dot {
            width: 14px; /* Reduced from 15px */
            height: 14px; /* Reduced from 15px */
            border: 2px solid white;
            border-radius: 50%;
        }
        
        .passcode-dot.filled {
            background-color: white;
        }
        
        .passcode-keypad {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            grid-gap: 15px;
            padding: 0 20px;
            width: 85%;
            max-width: 300px;
            margin: 0 auto;
        }
        
        .passcode-key {
            width: 65px;
            height: 65px;
            border-radius: 50%;
            background-color: rgba(255, 255, 255, 0.2);
            display: flex;
            justify-content: center;
            align-items: center;
            color: white;
            font-size: 28px;
            cursor: pointer;
            user-select: none;
            margin: 0 auto;
        }
        
        /* Style for the zero key */
        .passcode-key-zero {
            grid-column: 2;
        }
        
        .passcode-key:active {
            background-color: rgba(255, 255, 255, 0.4);
        }
        
        /* Home Screen */
        .home-screen {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: url('https://images.pexels.com/photos/3617500/pexels-photo-3617500.jpeg');
            background-size: cover;
            background-position: center;
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            grid-template-rows: repeat(5, 1fr);
            padding: 35px 10px 10px 10px;
            grid-gap: 5px;
            z-index: 10;
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.3s ease;
        }
        
        .home-screen.active {
            opacity: 1;
            pointer-events: all;
        }
        
        /* App Icon */
        .app-icon {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: transform 0.1s;
        }
        
        .app-icon:active {
            transform: scale(0.9);
        }
        
        .app-icon-img {
            width: 44px; /* Reduced from 54px */
            height: 44px; /* Reduced from 54px */
            border-radius: 10px; /* Reduced from 12px */
            margin-bottom: 4px; /* Reduced from 5px */
            display: flex;
            justify-content: center;
            align-items: center;
            font-size: 22px; /* Reduced from 26px */
            color: white;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.2);
        }
        
        .app-icon-label {
            font-size: 10px; /* Reduced from 11px */
            color: white;
            text-shadow: 0 1px 3px rgba(0, 0, 0, 0.5);
        }
        
        /* Dock */
        .dock {
            position: absolute;
            bottom: 15px; /* Reduced from 20px */
            left: 0;
            width: 100%;
            height: 60px; /* Reduced from 70px */
            background-color: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            border-radius: 18px; /* Reduced from 20px */
            margin: 0 10px;
            display: flex;
            justify-content: space-evenly;
            align-items: center;
        }
        
        /* App Screens */
        .app-screen {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: var(--bg-color);
            z-index: 15;
            transform: translateY(100%);
            transition: transform 0.3s cubic-bezier(0.19, 1, 0.22, 1);
            display: flex;
            flex-direction: column;
            padding-top: 20px; /* Added padding for status bar */
        }
        
        .app-screen.active {
            transform: translateY(0);
        }
        
        .app-header {
            height: 40px; /* Reduced from 44px */
            display: flex;
            justify-content: center;
            align-items: center;
            background-color: rgba(30, 30, 30, 0.8);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            position: relative;
            border-bottom: 1px solid var(--border-color);
        }
        
        .app-title {
            font-size: 16px; /* Reduced from 18px */
            font-weight: 500;
            color: white;
        }
        
        .app-content {
            flex: 1;
            overflow-y: auto;
            padding: 15px;
            color: white;
            display: flex;
            flex-direction: column;
        }
        
        /* Notes App Specific Styling */
        .notes-app {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: var(--bg-color);
            z-index: 15;
            transform: translateY(100%);
            transition: transform 0.3s cubic-bezier(0.19, 1, 0.22, 1);
            display: flex;
            flex-direction: column;
            padding-top: 20px; /* Added padding for status bar */
        }
        
        .notes-app.active {
            transform: translateY(0);
        }
        
        .notes-container {
            flex: 1;
            overflow: hidden;
            display: flex;
            flex-direction: column;
        }
        
        /* Notes Sidebar */
        .sidebar {
            flex: 1;
            background-color: var(--sidebar-bg);
            overflow-y: auto;
            transition: height 0.3s ease;
            position: relative;
        }
        
        .sidebar-header {
            padding: 15px;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            background-color: var(--sidebar-bg);
            z-index: 10;
        }
        
        .sidebar-header h1 {
            font-size: 20px; /* Reduced from 22px */
            font-weight: 600;
            color: var(--text-color);
        }
        
        .note-count {
            color: var(--secondary-text);
            font-size: 14px;
            margin-left: 5px;
        }
        
        .notes-list {
            list-style: none;
        }
        
        .note-item {
            padding: 12px;
            border-bottom: 1px solid var(--border-color);
            cursor: pointer;
            transition: background-color 0.2s;
        }
        
        .note-item:hover {
            background-color: rgba(255, 255, 255, 0.05);
        }
        
        .note-item.active {
            background-color: rgba(255, 149, 0, 0.1);
            border-left: 3px solid var(--highlight-color);
        }
        
        .note-title {
            font-weight: 600;
            margin-bottom: 5px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            color: var(--text-color);
        }
        
        .note-preview {
            color: var(--secondary-text);
            font-size: 14px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .note-date {
            font-size: 12px;
            color: var(--secondary-text);
            margin-top: 5px;
        }
        
        /* Note Detail View */
        .main-content {
            height: 0;
            overflow-y: auto;
            background-color: var(--note-bg);
            transition: height 0.3s ease;
            display: none;
        }
        
        .main-content.active {
            flex: 1;
            height: 100%;
            display: block;
        }
        
        .note-header {
            padding: 15px;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            background-color: var(--note-bg);
            z-index: 5;
            color: var(--text-color);
        }
        
        .note-content {
            padding: 15px;
            line-height: 1.6;
            white-space: pre-wrap;
            color: var(--text-color);
        }
        
        .note-content p {
            margin-bottom: 15px;
        }
        
        .back-to-list {
            background: none;
            border: none;
            color: var(--highlight-color);
            font-size: 16px;
            cursor: pointer;
            padding: 5px;
        }
        
        /* Generic App Content Styling */
        .app-placeholder {
            flex: 1;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            text-align: center;
            padding: 20px;
        }
        
        .app-placeholder-icon {
            font-size: 40px; /* Reduced from 48px */
            margin-bottom: 20px;
        }
        
        .app-placeholder-text {
            font-size: 16px; /* Reduced from 18px */
            color: #aaa;
        }
        
        /* Loading Screen */
        .loading {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: black;
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 30;
            opacity: 1;
            transition: opacity 1s ease;
        }
        
        .loading.hidden {
            opacity: 0;
            pointer-events: none;
        }
        
        .apple-logo {
            width: 70px; /* Reduced from 80px */
            opacity: 0.8;
        }
        
        /* Shake animation for wrong passcode */
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            10%, 30%, 50%, 70%, 90% { transform: translateX(-10px); }
            20%, 40%, 60%, 80% { transform: translateX(10px); }
        }
        
        /* Styles for the email app */
        .email-app-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            background-color: var(--email-bg);
            overflow: hidden;
        }
        
        .email-list {
            flex: 1;
            overflow-y: auto;
        }
        
        .email-item {
            padding: 12px 15px;
            border-bottom: 1px solid var(--border-color);
            cursor: pointer;
        }
        
        .email-item.unread {
            background-color: rgba(255, 255, 255, 0.05);
        }
        
        .email-item-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 5px;
        }
        
        .email-sender {
            font-weight: 600;
            color: var(--text-color);
        }
        
        .email-date {
            font-size: 12px;
            color: var(--secondary-text);
        }
        
        .email-subject {
            font-weight: 500;
            margin-bottom: 4px;
            color: var(--text-color);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .email-preview {
            font-size: 13px;
            color: var(--secondary-text);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .email-detail {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: var(--email-bg);
            z-index: 20;
            display: flex;
            flex-direction: column;
            transform: translateX(100%);
            transition: transform 0.3s ease;
        }
        
        .email-detail.active {
            transform: translateX(0);
        }
        
        .email-detail-header {
            padding: 15px;
            border-bottom: 1px solid var(--border-color);
            background-color: var(--email-header-bg);
        }
        
        .email-detail-back {
            background: none;
            border: none;
            color: var(--email-accent);
            font-size: 24px;
            cursor: pointer;
            padding: 0;
            margin-bottom: 10px;
        }
        
        .email-detail-subject {
            font-size: 18px;
            font-weight: 600;
            color: var(--email-text-color);
            margin-bottom: 5px;
        }
        
        .email-detail-info {
            display: flex;
            justify-content: space-between;
            margin-bottom: 10px;
        }
        
        .email-detail-sender {
            color: var(--email-text-color);
        }
        
        .email-detail-date {
            font-size: 12px;
            color: var(--secondary-text);
        }
        
        .email-detail-content {
            flex: 1;
            padding: 15px;
            overflow-y: auto;
            color: var(--email-text-color);
            line-height: 1.5;
        }
        
        .email-detail-content p {
            margin-bottom: 15px;
        }
        
        /* Styles for the messaging app */
        .message-app-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            background-color: var(--bg-color);
            overflow: hidden;
        }
        
        .chat-list {
            flex: 1;
            overflow-y: auto;
        }
        
        .chat-item {
            padding: 12px 15px;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            cursor: pointer;
        }
        
        .chat-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: var(--border-color);
            display: flex;
            justify-content: center;
            align-items: center;
            color: var(--text-color);
            font-weight: bold;
            margin-right: 10px;
        }
        
        .chat-info {
            flex: 1;
        }
        
        .chat-name {
            font-weight: 600;
            color: var(--text-color);
            margin-bottom: 3px;
        }
        
        .chat-last-message {
            font-size: 13px;
            color: var(--secondary-text);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .chat-time {
            font-size: 12px;
            color: var(--secondary-text);
            margin-left: 10px;
        }
        
        .chat-detail {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: var(--bg-color);
            z-index: 20;
            display: flex;
            flex-direction: column;
            transform: translateX(100%);
            transition: transform 0.3s ease;
            padding-top: 20px;
        }
        
        .chat-detail.active {
            transform: translateX(0);
        }
        
        .chat-header {
            height: 44px;
            display: flex;
            align-items: center;
            padding: 0 15px;
            background-color: rgba(30, 30, 30, 0.9);
            border-bottom: 1px solid var(--border-color);
        }
        
        .chat-back {
            background: none;
            border: none;
            color: var(--highlight-color);
            font-size: 24px;
            cursor: pointer;
            padding: 0;
            margin-right: 10px;
        }
        
        .chat-title {
            font-size: 16px;
            font-weight: 600;
            color: var(--text-color);
            flex: 1;
            text-align: center;
        }
        
        .chat-option {
            background: none;
            border: none;
            color: var(--highlight-color);
            font-size: 24px;
            cursor: pointer;
            padding: 0;
        }
        
        .messages-container {
            flex: 1;
            overflow-y: auto;
            padding: 10px;
            display: flex;
            flex-direction: column;
        }
        
        .date-divider {
            text-align: center;
            color: var(--secondary-text);
            font-size: 12px;
            margin: 10px 0;
            position: relative;
        }
        
        .date-divider::before {
            content: '';
            position: absolute;
            left: 0;
            top: 50%;
            width: 40%;
            height: 1px;
            background-color: var(--border-color);
        }
        
        .date-divider::after {
            content: '';
            position: absolute;
            right: 0;
            top: 50%;
            width: 40%;
            height: 1px;
            background-color: var(--border-color);
        }
        
        .message {
            max-width: 70%;
            margin-bottom: 8px;
            padding: 8px 12px;
            border-radius: 18px;
            line-height: 1.4;
            position: relative;
        }
        
        .message.sent {
            align-self: flex-end;
            background-color: var(--message-sent-bg);
            color: white;
            border-bottom-right-radius: 4px;
        }
        
        .message.received {
            align-self: flex-start;
            background-color: var(--message-received-bg);
            color: white;
            border-bottom-left-radius: 4px;
        }
        
        .message-time {
            font-size: 10px;
            margin-top: 4px;
            opacity: 0.8;
            text-align: right;
        }
        
        .message-input-container {
            display: flex;
            align-items: center;
            padding: 10px;
            border-top: 1px solid var(--border-color);
        }
        
        .message-input {
            flex: 1;
            height: 36px;
            border-radius: 18px;
            border: none;
            background-color: #333;
            color: white;
            padding: 0 12px;
        }
        
        .send-button {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            border: none;
            background-color: var(--highlight-color);
            color: white;
            margin-left: 10px;
            display: flex;
            justify-content: center;
            align-items: center;
            font-size: 18px;
        }
        
        .read-receipt {
            font-size: 10px;
            color: var(--secondary-text);
            text-align: right;
            margin-top: 2px;
        }
        
        /* Styles for notification badge */
        .notification-badge {
            position: absolute;
            top: -5px;
            right: -5px;
            width: 18px;
            height: 18px;
            background-color: red;
            border-radius: 50%;
            color: white;
            font-size: 10px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        /* Styles for the gallery app */
        .gallery-app-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            background-color: black;
            overflow: hidden;
        }
        
        .gallery-grid {
            flex: 1;
            overflow-y: auto;
            padding: 2px;
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            grid-gap: 2px;
        }
        
        .gallery-item {
            aspect-ratio: 1/1;
            background-color: #222;
            background-position: center;
            background-size: cover;
            position: relative;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .gallery-item:active {
            transform: scale(0.97);
        }
        
        .image-detail {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: black;
            z-index: 20;
            display: flex;
            flex-direction: column;
            transform: translateX(100%);
            transition: transform 0.3s ease;
        }
        
        .image-detail.active {
            transform: translateX(0);
        }
        
        .image-detail-header {
            height: 44px;
            display: flex;
            align-items: center;
            padding: 0 15px;
            background-color: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            z-index: 2;
        }
        
        .image-detail-back {
            background: none;
            border: none;
            color: var(--highlight-color);
            font-size: 24px;
            cursor: pointer;
            padding: 0;
            margin-right: 10px;
        }
        
        .image-detail-title {
            font-size: 16px;
            font-weight: 600;
            color: white;
        }
        
        .image-detail-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            position: relative;
        }
        
        .detail-image {
            width: 100%;
            height: 100%;
            object-fit: contain;
            background-color: black;
        }
        
        .image-detail-info {
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            padding: 15px;
            background-color: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(10px);
            -webkit-backdrop-filter: blur(10px);
        }
        
        .image-detail-date {
            font-size: 12px;
            color: #999;
            margin-bottom: 5px;
        }
        
        .image-detail-description {
            font-size: 14px;
            color: white;
            line-height: 1.4;
        }
        
        /* Updated styles for the Phone app */
        .phone-app-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            background-color: var(--bg-color);
            overflow: hidden;
        }
        
        .call-log {
            flex: 1;
            overflow-y: auto;
        }
        
        .call-log-item {
            padding: 12px 15px;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            align-items: center;
            cursor: pointer;
        }
        
        .call-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: var(--border-color);
            display: flex;
            justify-content: center;
            align-items: center;
            color: var(--text-color);
            font-weight: bold;
            margin-right: 10px;
        }
        
        .call-outgoing {
            color: #4CD964;
        }
        
        .call-incoming {
            color: #007AFF;
        }
        
        .call-missed {
            color: #FF3B30;
        }
        
        .call-info-preview {
            flex: 1;
        }
        
        .call-name {
            font-weight: 600;
            color: var(--text-color);
            margin-bottom: 3px;
        }
        
        .call-number {
            font-size: 13px;
            color: var(--secondary-text);
        }
        
        .call-time-preview {
            font-size: 12px;
            color: var(--secondary-text);
            margin-left: 10px;
        }
        
        .call-detail {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: var(--bg-color);
            z-index: 20;
            display: flex;
            flex-direction: column;
            transform: translateX(100%);
            transition: transform 0.3s ease;
            padding-top: 20px;
        }
        
        .call-detail.active {
            transform: translateX(0);
        }
        
        .call-detail-header {
            height: 44px;
            display: flex;
            align-items: center;
            padding: 0 15px;
            background-color: rgba(30, 30, 30, 0.9);
            border-bottom: 1px solid var(--border-color);
        }
        
        .call-detail-back {
            background: none;
            border: none;
            color: var(--highlight-color);
            font-size: 24px;
            cursor: pointer;
            padding: 0;
            margin-right: 10px;
        }
        
        .call-detail-title {
            font-size: 16px;
            font-weight: 600;
            color: var(--text-color);
        }
        
        .call-detail-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            padding: 20px;
            align-items: center;
            overflow-y: auto;
        }
        
        .call-info {
            width: 100%;
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .call-contact-avatar {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background-color: var(--border-color);
            display: flex;
            justify-content: center;
            align-items: center;
            color: var(--text-color);
            font-size: 32px;
            font-weight: bold;
            margin-bottom: 15px;
        }
        
        .call-contact-name {
            font-size: 24px;
            font-weight: 600;
            color: var(--text-color);
            margin-bottom: 5px;
            text-align: center;
        }
        
        .call-contact-number {
            font-size: 16px;
            color: var(--secondary-text);
            margin-bottom: 15px;
            text-align: center;
        }
        
        .call-datetime, .call-duration, .call-type {
            font-size: 14px;
            color: var(--secondary-text);
            margin-bottom: 8px;
            text-align: center;
        }
        
        .call-audio-player {
            width: 100%;
            margin-top: 25px;
            background-color: rgba(60, 60, 60, 0.3);
            border-radius: 10px;
            padding: 10px;
        }
        
        .audio-player {
            width: 100%;
            height: 40px;
        }
        
        .call-description {
            margin-top: 20px;
            padding: 15px;
            background-color: rgba(60, 60, 60, 0.3);
            border-radius: 10px;
            color: var(--text-color);
            font-size: 14px;
            line-height: 1.5;
            width: 100%;
        }
    </style>
    
    <!-- Direct styles for mobile that will definitely be applied -->
    <style type="text/css">
        @media screen and (max-width: 768px) {
            /* Larger numeric keypad for mobile */
            .passcode-key {
                width: 70px !important;
                height: 70px !important;
                font-size: 32px !important;
            }
            
            .passcode-keypad {
                grid-gap: 20px !important;
                width: 90% !important;
                max-width: 350px !important;
                margin-top: 25px !important;
            }
            
            .passcode-dots {
                gap: 20px !important;
                margin-bottom: 35px !important;
            }
            
            .passcode-dot {
                width: 16px !important;
                height: 16px !important;
            }
        }
            /* Complete reset for mobile devices */
            html, body {
                width: 100% !important;
                height: 100% !important;
                margin: 0 !important;
                padding: 0 !important;
                overflow: hidden !important;
                position: fixed !important;
                background: #000 !important;
            }
            
            /* Full-screen iPhone */
            .iphone-container {
                position: fixed !important;
                top: 0 !important;
                left: 0 !important;
                right: 0 !important;
                bottom: 0 !important;
                width: 100% !important;
                height: 100% !important;
                max-width: none !important;
                max-height: none !important;
                margin: 0 !important;
                padding: 0 !important;
                border: none !important;
                border-radius: 0 !important;
                box-shadow: none !important;
                transform: none !important;
                overflow: hidden !important;
            }
            
            /* Full-screen display */
            .iphone-screen {
                top: 30px !important;
                height: calc(100% - 60px) !important;
            }
            
            /* Small bezels */
            .iphone-top-bezel {
                height: 30px !important;
            }
            
            .iphone-bottom-bezel {
                height: 30px !important;
            }
        }
    </style>
</head>
<body>
    <div class="iphone-container">
        <!-- iPhone 5s Hardware Elements -->
        <div class="iphone-top-bezel">
            <div class="iphone-speaker"></div>
            <div class="iphone-camera">
                <div class="camera-lens"></div>
                <div class="camera-lens"></div>
                <div class="camera-flash"></div>
            </div>
        </div>
        <div class="iphone-bottom-bezel">
            <div class="iphone-home-button" id="home-button">
                <div class="iphone-home-button-square"></div>
            </div>
        </div>
        <div class="iphone7-power-button"></div>
        <div class="iphone7-volume-up"></div>
        <div class="iphone7-volume-down"></div>
        <div class="iphone7-mute-switch"></div>
        
        <!-- Loading Screen (Apple Logo) -->
        <div class="loading" id="loading">
            <svg class="apple-logo" viewBox="0 0 170 170" xmlns="http://www.w3.org/2000/svg">
                <path fill="white" d="M150.37 130.25c-2.45 5.66-5.35 10.87-8.71 15.66-4.58 6.53-8.33 11.05-11.22 13.56-4.48 4.12-9.28 6.23-14.42 6.35-3.69 0-8.14-1.05-13.32-3.18-5.197-2.12-9.973-3.17-14.34-3.17-4.58 0-9.492 1.05-14.746 3.17-5.262 2.13-9.501 3.24-12.742 3.35-4.929 0.21-9.842-1.96-14.746-6.52-3.13-2.73-7.045-7.41-11.735-14.04-5.032-7.08-9.169-15.29-12.41-24.65-3.471-10.11-5.211-19.9-5.211-29.378 0-10.857 2.346-20.221 7.045-28.068 3.693-6.303 8.606-11.275 14.755-14.925s12.793-5.51 19.948-5.629c3.915 0 9.049 1.211 15.429 3.591 6.362 2.388 10.447 3.599 12.238 3.599 1.339 0 5.877-1.416 13.57-4.239 7.275-2.618 13.415-3.702 18.445-3.275 13.63 1.1 23.87 6.473 30.68 16.153-12.19 7.386-18.22 17.731-18.1 31.002 0.11 10.337 3.86 18.939 11.23 25.769 3.34 3.17 7.07 5.62 11.22 7.36-0.9 2.61-1.85 5.11-2.86 7.51zM119.11 7.24c0 8.102-2.96 15.667-8.86 22.669-7.12 8.324-15.732 13.134-25.071 12.375-0.119-0.972-0.188-1.995-0.188-3.07 0-7.778 3.386-16.102 9.399-22.908 3.002-3.446 6.82-6.311 11.45-8.597 4.62-2.252 8.99-3.497 13.1-3.71 0.12 1.083 0.17 2.166 0.17 3.24z"/>
            </svg>
        </div>
        
        <div class="iphone-screen">
            <!-- Lock Screen -->
            <div class="lock-screen" id="lock-screen">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="lock-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="lock-time" id="lock-time">9:41</div>
                <div class="lock-date" id="lock-date">Sunday, March 9</div>
                
                <!-- Improved slider like in the image -->
                <div class="slide-to-unlock" id="slide-to-unlock">
                    <div class="slide-arrow" id="slide-arrow"></div>
                    <div class="slide-text">slide to unlock</div>
                </div>
                <div class="camera-icon">📷</div>
            </div>
            
            <!-- Passcode Screen -->
            <div class="passcode-screen" id="passcode-screen">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="passcode-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="passcode-title">Enter passcode</div>
                <div class="passcode-dots">
                    <div class="passcode-dot" id="dot-1"></div>
                    <div class="passcode-dot" id="dot-2"></div>
                    <div class="passcode-dot" id="dot-3"></div>
                    <div class="passcode-dot" id="dot-4"></div>
                </div>
                <div class="passcode-keypad">
    <div class="passcode-key" data-key="1">1</div>
    <div class="passcode-key" data-key="2">2</div>
    <div class="passcode-key" data-key="3">3</div>
    <div class="passcode-key" data-key="4">4</div>
    <div class="passcode-key" data-key="5">5</div>
    <div class="passcode-key" data-key="6">6</div>
    <div class="passcode-key" data-key="7">7</div>
    <div class="passcode-key" data-key="8">8</div>
    <div class="passcode-key" data-key="9">9</div>
    <div class="passcode-key passcode-key-zero" data-key="0">0</div>
</div>
            </div>
            
            <!-- Home Screen with Gallery and Phone icons updated -->
            <div class="home-screen" id="home-screen">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="home-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                
                <!-- App Icons -->
                <div class="app-icon" id="notes-icon">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #FFD60A, #FFC30A);">📝</div>
                    <div class="app-icon-label">Notes</div>
                </div>
                
                <div class="app-icon" id="phone-icon" style="position: relative;">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #5E5CE6, #3634A3);">📱</div>
                    <div class="notification-badge">2</div>
                    <div class="app-icon-label">Phone</div>
                </div>
                
                <div class="app-icon" id="messages-icon" style="position: relative;">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #33C759, #2A9D48);">💬</div>
                    <div class="notification-badge">1</div>
                    <div class="app-icon-label">Messages</div>
                </div>
                
                <div class="app-icon" id="mail-icon" style="position: relative;">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #FF3A2D, #C1291F);">📧</div>
                    <div class="notification-badge">2</div>
                    <div class="app-icon-label">Mail</div>
                </div>
                
                <div class="app-icon" id="photos-icon">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #FCBC00, #FD9500);">🖼️</div>
                    <div class="notification-badge">3</div>
                    <div class="app-icon-label">Photos</div>
                </div>
                
                <div class="app-icon" id="music-icon">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #FC3158, #CE2549);">🎵</div>
                    <div class="app-icon-label">Music</div>
                </div>
                
                <div class="app-icon" id="weather-icon">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #5AC8FA, #4A9CC9);">☁️</div>
                    <div class="app-icon-label">Weather</div>
                </div>
                
                <div class="app-icon" id="calendar-icon">
                    <div class="app-icon-img" style="background: linear-gradient(135deg, #FF9500, #C77700);">📅</div>
                    <div class="app-icon-label">Calendar</div>
                </div>
                
                <!-- Dock at the bottom -->
                <div class="dock">
                    <div class="app-icon">
                        <div class="app-icon-img" style="background: linear-gradient(135deg, #4CD964, #36AC4C);">📞</div>
                    </div>
                    <div class="app-icon">
                        <div class="app-icon-img" style="background: linear-gradient(135deg, #30B0C7, #239DAF);">🌐</div>
                    </div>
                    <div class="app-icon">
                        <div class="app-icon-img" style="background: linear-gradient(135deg, #FF2D54, #CC2447);">📷</div>
                    </div>
                    <div class="app-icon">
                        <div class="app-icon-img" style="background: linear-gradient(135deg, #AF52DE, #8641AE);">⚙️</div>
                    </div>
                </div>
            </div>
            
            <!-- Phone App with Audio Playback -->
            <div class="app-screen" id="phone-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="phone-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Phone</div>
                </div>
                <div class="phone-app-content">
                    <div class="call-log" id="call-log">
                        <!-- Call log items will be populated by JavaScript -->
                    </div>
                    
                    <!-- Call Detail View Template (will be cloned for each call) -->
                    <div class="call-detail" id="call-detail-template">
                        <div class="status-bar">
                            <div class="status-carrier">Vivo</div>
                            <div class="status-time">9:41</div>
                            <div class="status-icons">
                                <span>📶</span>
                                <span>🔋</span>
                            </div>
                        </div>
                        <div class="call-detail-header">
                            <button class="call-detail-back">←</button>
                            <div class="call-detail-title">Call Details</div>
                        </div>
                        <div class="call-detail-content">
                            <div class="call-info">
                                <div class="call-contact-avatar"></div>
                                <div class="call-contact-name"></div>
                                <div class="call-contact-number"></div>
                                <div class="call-datetime"></div>
                                <div class="call-duration"></div>
                                <div class="call-type"></div>
                            </div>
                            <div class="call-audio-player">
                                <audio class="audio-player" controls>
                                    <source src="" type="audio/mpeg">
                                    Your browser does not support HTML5 audio.
                                </audio>
                            </div>
                            <div class="call-description"></div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Photos/Gallery App -->
            <div class="app-screen" id="photos-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="photos-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Photos</div>
                </div>
                <div class="gallery-app-content">
                    <div class="gallery-grid" id="gallery-grid">
                        <!-- Gallery items will be populated by JavaScript -->
                    </div>
                    
                    <!-- Image Detail View Template (will be cloned for each image) -->
                    <div class="image-detail" id="image-detail-template">
                        <div class="status-bar">
                            <div class="status-carrier">Vivo</div>
                            <div class="status-time">9:41</div>
                            <div class="status-icons">
                                <span>📶</span>
                                <span>🔋</span>
                            </div>
                        </div>
                        <div class="image-detail-header">
                            <button class="image-detail-back">←</button>
                            <div class="image-detail-title"></div>
                        </div>
                        <div class="image-detail-content">
                            <img class="detail-image" src="" alt="">
                            <div class="image-detail-info">
                                <div class="image-detail-date"></div>
                                <div class="image-detail-description"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Messages App -->
            <div class="app-screen" id="messages-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="messages-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Messages</div>
                </div>
                <div class="message-app-content">
                    <div class="chat-list">
                        <div class="chat-item" id="unknown-chat">
                            <div class="chat-avatar">?</div>
                            <div class="chat-info">
                                <div class="chat-name">Unknown</div>
                                <div class="chat-last-message">You should have listened. It's too late now.</div>
                            </div>
                            <div class="chat-time">Today</div>
                        </div>
                        <!-- Other chats here -->
                    </div>
                    
                    <!-- Chat Detail View -->
                    <div class="chat-detail" id="unknown-chat-detail">
                        <div class="status-bar">
                            <div class="status-carrier">Vivo</div>
                            <div class="status-time" id="chat-detail-status-time">9:41</div>
                            <div class="status-icons">
                                <span>📶</span>
                                <span>🔋</span>
                            </div>
                        </div>
                        <div class="chat-header">
                            <button class="chat-back" id="chat-back">←</button>
                            <div class="chat-title">Unknown</div>
                            <button class="chat-option">ⓘ</button>
                        </div>
                        <div class="messages-container" id="unknown-messages">
                            <!-- Messages will be added via JavaScript -->
                        </div>
                        <div class="message-input-container">
                            <input type="text" class="message-input" placeholder="Message">
                            <button class="send-button">↑</button>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Mail App -->
            <div class="app-screen" id="mail-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="mail-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Mail</div>
                </div>
                <div class="email-app-content">
                    <div class="email-list">
                        <div class="email-item" id="email-yale-response">
                            <div class="email-item-header">
                                <div class="email-sender">Yale University</div>
                                <div class="email-date">Jan 27</div>
                            </div>
                            <div class="email-subject">Response to Request - Academic Records Query</div>
                            <div class="email-preview">Dear Ms. Collins, In response to your request sent on January 20, 2024...</div>
                        </div>
                        <div class="email-item" id="email-yale-request">
                            <div class="email-item-header">
                                <div class="email-sender">To: Yale University</div>
                                <div class="email-date">Jan 20</div>
                            </div>
                            <div class="email-subject">Urgent academic confirmation request</div>
                            <div class="email-preview">Dear Sir/Madam, My name is Emma Collins, I am an investigative journalist currently...</div>
                        </div>
                        <!-- Other emails here -->
                    </div>
                    
                    <!-- Email Detail View - Yale Response -->
                    <div class="email-detail" id="yale-response-detail">
                        <div class="status-bar">
                            <div class="status-carrier">Vivo</div>
                            <div class="status-time">9:41</div>
                            <div class="status-icons">
                                <span>📶</span>
                                <span>🔋</span>
                            </div>
                        </div>
                        <div class="email-detail-header">
                            <button class="email-detail-back" id="email-back-1">←</button>
                            <div class="email-detail-subject">Response to Request - Academic Records Query</div>
                            <div class="email-detail-info">
                                <div class="email-detail-sender">From: Yale University &lt;richard.evans@yale.edu&gt;</div>
                                <div class="email-detail-date">Jan 27, 2024, 11:43</div>
                            </div>
                        </div>
                        <div class="email-detail-content">
                            <h3 style="color: #4dabf7; margin-bottom: 5px;">YALE UNIVERSITY</h3>
                            <p style="color: #adb5bd; margin-bottom: 15px;">Department of Admissions and Academic Records</p>
                            <p>Date: January 27, 2024<br>
                            To: Emma Collins (emmacollins.journal@gmail.com)</p>
                            
                            <p>Dear Ms. Collins,</p>
                            
                            <p>In response to your request sent on January 20, 2024, we inform you that after a thorough search of our academic records, we have found no evidence or record of enrollment, attendance or graduation in the name of Sarah Elizabeth Mitchell, born on April 15, 2000.</p>
                            
                            <p>We emphasize that all academic records at Yale University are meticulously archived and maintained in complete security. Therefore, we can state with full certainty that the person referred to has never been enrolled nor taken any courses at this institution.</p>
                            
                            <p>For additional clarification, we remain at your disposal.</p>
                            
                            <div style="margin-top: 25px; padding-top: 15px; border-top: 1px solid #333;">
                                <p>Sincerely,</p>
                                <p style="font-weight: bold;">Richard K. Evans</p>
                                <p style="color: #adb5bd;">General Registrar<br>
                                Yale University</p>
                            </div>
                            
                            <div style="margin-top: 25px; font-size: 12px; color: #6c757d;">
                                <p>Yale University | New Haven, Connecticut 06520 | (203) 432-4771</p>
                                <p style="font-style: italic;">This message contains confidential information and is intended exclusively for the recipient mentioned above.</p>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Email Detail View - Yale Request -->
                    <div class="email-detail" id="yale-request-detail">
                        <div class="status-bar">
                            <div class="status-carrier">Vivo</div>
                            <div class="status-time">9:41</div>
                            <div class="status-icons">
                                <span>📶</span>
                                <span>🔋</span>
                            </div>
                        </div>
                        <div class="email-detail-header">
                            <button class="email-detail-back" id="email-back-2">←</button>
                            <div class="email-detail-subject">Urgent academic confirmation request</div>
                            <div class="email-detail-info">
                                <div class="email-detail-sender">To: Department of Academic Records &lt;registrar@yale.edu&gt;</div>
                                <div class="email-detail-date">Jan 20, 2024, 09:15</div>
                            </div>
                        </div>
                        <div class="email-detail-content">
                            <p>Dear Sir/Madam,</p>
                            
                            <p>My name is Emma Collins, I am an investigative journalist currently working on a report for the Ravenwood Times newspaper. I would like to request an official verification of the academic history of Sarah Elizabeth Mitchell, born on April 15, 2000. The mentioned individual publicly claims to have attended and graduated from Yale University between 2018 and 2022.</p>
                            
                            <p>This information is crucial for the accuracy and integrity of the investigative article I am developing, therefore, I would kindly ask you to confirm whether Sarah Elizabeth Mitchell has or does not have any academic connection with your institution.</p>
                            
                            <p>I await an official response, and thank you in advance for your attention and cooperation.</p>
                            
                            <div style="margin-top: 25px;">
                                <p>Sincerely,</p>
                                <p style="font-weight: bold;">Emma Collins</p>
                                <p>Investigative Journalist<br>
                                Ravenwood Times</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Safari App -->
            <div class="app-screen" id="safari-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="safari-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Safari</div>
                </div>
                <div class="app-content">
                    <div class="app-placeholder">
                        <div class="app-placeholder-icon">🌐</div>
                        <div class="app-placeholder-text">Here you would browse the internet.</div>
                    </div>
                </div>
            </div>
            
            <!-- Music App -->
            <div class="app-screen" id="music-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="music-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Music</div>
                </div>
                <div class="app-content">
                    <div class="app-placeholder">
                        <div class="app-placeholder-icon">🎵</div>
                        <div class="app-placeholder-text">Your music library would be here.</div>
                    </div>
                </div>
            </div>
            
            <!-- Weather App -->
            <div class="app-screen" id="weather-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="weather-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Weather</div>
                </div>
                <div class="app-content">
                    <div class="app-placeholder">
                        <div class="app-placeholder-icon">☁️</div>
                        <div class="app-placeholder-text">Weather information would appear here.</div>
                    </div>
                </div>
            </div>
            
            <!-- Calendar App -->
            <div class="app-screen" id="calendar-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="calendar-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                <div class="app-header">
                    <div class="app-title">Calendar</div>
                </div>
                <div class="app-content">
                    <div class="app-placeholder">
                        <div class="app-placeholder-icon">📅</div>
                        <div class="app-placeholder-text">Your events and appointments would be here.</div>
                    </div>
                </div>
            </div>
            
            <!-- Notes App -->
            <div class="notes-app" id="notes-app">
                <div class="status-bar">
                    <div class="status-carrier">Vivo</div>
                    <div class="status-time" id="notes-status-time">9:41</div>
                    <div class="status-icons">
                        <span>📶</span>
                        <span>🔋</span>
                    </div>
                </div>
                
                <div class="app-header">
                    <div class="app-title">Notes <span class="note-count" id="note-count"></span></div>
                </div>
                
                <div class="notes-container" id="notes-container">
                    <aside class="sidebar" id="sidebar">
                        <ul class="notes-list" id="notes-list">
                            <!-- Notes will be populated here by JavaScript -->
                        </ul>
                    </aside>
                    
                    <main class="main-content" id="main-content">
                        <div class="note-header" id="note-header">
                            <button class="back-to-list" id="back-to-list">←</button>
                            <div class="note-title-display"></div>
                        </div>
                        <div class="note-content" id="note-content">
                            <p>Select a note to view its content.</p>
                        </div>
                    </main>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Load notes from PHP
        const notes = <?php echo json_encode($notes); ?>;
        
        // Load gallery data
        const gallery = <?php echo json_encode($gallery); ?>;
        
        // Load phone calls data
        const phones = <?php echo json_encode($phones); ?>;
        
        // DOM Elements
        const notesList = document.getElementById('notes-list');
        const noteContent = document.getElementById('note-content');
        const noteCount = document.getElementById('note-count');
        const notesApp = document.getElementById('notes-app');
        const sidebar = document.getElementById('sidebar');
        const mainContent = document.getElementById('main-content');
        const backToList = document.getElementById('back-to-list');
        const loading = document.getElementById('loading');
        const homeButton = document.getElementById('home-button');
        const lockScreen = document.getElementById('lock-screen');
        const passcodeScreen = document.getElementById('passcode-screen');
        const homeScreen = document.getElementById('home-screen');
        const slideToUnlock = document.getElementById('slide-to-unlock');
        const slideArrow = document.getElementById('slide-arrow');
        const passcodeKeys = document.querySelectorAll('.passcode-key');
        const passcodeAllDots = document.querySelectorAll('.passcode-dot');
        const appScreens = document.querySelectorAll('.app-screen');
        
        // App Icons
        const notesIcon = document.getElementById('notes-icon');
        const phoneIcon = document.getElementById('phone-icon');
        const messagesIcon = document.getElementById('messages-icon');
        const mailIcon = document.getElementById('mail-icon');
        const photosIcon = document.getElementById('photos-icon');
        const musicIcon = document.getElementById('music-icon');
        const weatherIcon = document.getElementById('weather-icon');
        const calendarIcon = document.getElementById('calendar-icon');
        
        // Email elements
        const emailYaleResponse = document.getElementById('email-yale-response');
        const emailYaleRequest = document.getElementById('email-yale-request');
        const yaleResponseDetail = document.getElementById('yale-response-detail');
        const yaleRequestDetail = document.getElementById('yale-request-detail');
        const emailBack1 = document.getElementById('email-back-1');
        const emailBack2 = document.getElementById('email-back-2');
        
        // Messages elements
        const unknownChat = document.getElementById('unknown-chat');
        const unknownChatDetail = document.getElementById('unknown-chat-detail');
        const chatBack = document.getElementById('chat-back');
        const unknownMessages = document.getElementById('unknown-messages');
        
        // Gallery elements
        const photosApp = document.getElementById('photos-app');
        const galleryGrid = document.getElementById('gallery-grid');
        const imageDetailTemplate = document.getElementById('image-detail-template');
        
        // Phone elements
        const phoneApp = document.getElementById('phone-app');
        const callLog = document.getElementById('call-log');
        const callDetailTemplate = document.getElementById('call-detail-template');
        
        // Time elements
        const allTimeElements = document.querySelectorAll('.status-time');
        const lockTime = document.getElementById('lock-time');
        const lockDate = document.getElementById('lock-date');
        
        // Current active app
        let currentApp = null;
        
        // Passcode variables
        let passcodeInput = '';
        const correctPasscode = '1612';

        // Update time in status bar
        function updateTime() {
            const now = new Date();
            const hours = now.getHours().toString().padStart(2, '0');
            const minutes = now.getMinutes().toString().padStart(2, '0');
            const timeString = `${hours}:${minutes}`;
            
            // Update all time elements
            allTimeElements.forEach(el => {
                el.textContent = timeString;
            });
            
            // Update lock screen time and date
            lockTime.textContent = timeString;
            
            const options = { weekday: 'long', day: 'numeric', month: 'long' };
            lockDate.textContent = now.toLocaleDateString('en-US', options);
        }

        // Update time every minute
        updateTime();
        setInterval(updateTime, 60000);
        
        // Hide loading screen after a delay
        setTimeout(() => {
            loading.classList.add('hidden');
        }, 2000);
        
        // Slide to unlock functionality with drag
        let isDragging = false;
        let startX, startLeft;
        
        slideArrow.addEventListener('mousedown', startDrag);
        slideArrow.addEventListener('touchstart', startDrag);
        
        document.addEventListener('mousemove', drag);
        document.addEventListener('touchmove', drag);
        
        document.addEventListener('mouseup', endDrag);
        document.addEventListener('touchend', endDrag);
        
        function startDrag(e) {
            isDragging = true;
            startX = e.type === 'mousedown' ? e.clientX : e.touches[0].clientX;
            startLeft = parseInt(window.getComputedStyle(slideArrow).left);
            e.preventDefault();
        }
        
        function drag(e) {
            if (!isDragging) return;
            
            const x = e.type === 'mousemove' ? e.clientX : e.touches[0].clientX;
            const deltaX = x - startX;
            const slideWidth = slideToUnlock.offsetWidth;
            
            // Calculate new position, but keep it within bounds
            let newLeft = Math.max(5, Math.min(slideWidth - slideArrow.offsetWidth - 5, startLeft + deltaX));
            slideArrow.style.left = `${newLeft}px`;
            
            // If slid more than 70% across, trigger unlock
            if (newLeft > (slideWidth * 0.7)) {
                endDrag();
                passcodeScreen.classList.add('active');
            }
        }
        
        function endDrag() {
            if (!isDragging) return;
            isDragging = false;
            
            // Reset slider position with animation
            slideArrow.style.transition = 'left 0.3s';
            slideArrow.style.left = '5px';
            
            // Remove transition after animation completes
            setTimeout(() => {
                slideArrow.style.transition = '';
            }, 300);
        }
        
        // Double-click on slider will also unlock (for ease of use)
        slideToUnlock.addEventListener('dblclick', () => {
            passcodeScreen.classList.add('active');
        });
        
        // Passcode input
        passcodeKeys.forEach(key => {
            key.addEventListener('click', () => {
                const keyValue = key.getAttribute('data-key');
                if (passcodeInput.length < 4) {
                    passcodeInput += keyValue;
                    updatePasscodeDots();
                    
                    // Check if passcode is complete
                    if (passcodeInput.length === 4) {
                        setTimeout(() => {
                            if (passcodeInput === correctPasscode) {
                                unlockPhone();
                            } else {
                                // Wrong passcode - shake effect
                                passcodeScreen.style.animation = 'shake 0.5s';
                                setTimeout(() => {
                                    passcodeScreen.style.animation = '';
                                    resetPasscode();
                                }, 500);
                            }
                        }, 300);
                    }
                }
            });
        });
        
        // Update passcode dots
        function updatePasscodeDots() {
            for (let i = 0; i < passcodeAllDots.length; i++) {
                if (i < passcodeInput.length) {
                    passcodeAllDots[i].classList.add('filled');
                } else {
                    passcodeAllDots[i].classList.remove('filled');
                }
            }
        }
        
        // Reset passcode input
        function resetPasscode() {
            passcodeInput = '';
            updatePasscodeDots();
        }
        
        // Unlock the phone
        function unlockPhone() {
            passcodeScreen.classList.remove('active');
            lockScreen.style.display = 'none';
            homeScreen.classList.add('active');
        }
        
        // Home button functionality
        homeButton.addEventListener('click', () => {
            // Close any open app
            if (currentApp) {
                currentApp.classList.remove('active');
                currentApp = null;
                
                // Reset details views
                yaleResponseDetail.classList.remove('active');
                yaleRequestDetail.classList.remove('active');
                unknownChatDetail.classList.remove('active');
                
                // Reset notes view
                sidebar.style.display = 'block';
                mainContent.classList.remove('active');
                
                // Reset any audio playing in call detail views
                document.querySelectorAll('.call-detail.active').forEach(elem => {
                    const audio = elem.querySelector('audio');
                    if (audio) audio.pause();
                    elem.classList.remove('active');
                });
                
                // Reset any open image detail views
                document.querySelectorAll('.image-detail.active').forEach(elem => {
                    elem.classList.remove('active');
                });
            }
            
            // If locked, show lock screen
            if (!homeScreen.classList.contains('active')) {
                lockScreen.style.display = 'flex';
                resetPasscode();
                passcodeScreen.classList.remove('active');
            }
        });
        
        // App opening logic
        function openApp(appElement) {
            // Close current app if any
            if (currentApp) {
                currentApp.classList.remove('active');
            }
            
            // Open new app
            appElement.classList.add('active');
            currentApp = appElement;
            
            // For message app, remove notification badge
            if (appElement.id === 'messages-app') {
                const badge = messagesIcon.querySelector('.notification-badge');
                if (badge) badge.style.display = 'none';
            }
            
            // For email app, remove notification badge
            if (appElement.id === 'mail-app') {
                const badge = mailIcon.querySelector('.notification-badge');
                if (badge) badge.style.display = 'none';
            }
            
            // For photos app, remove notification badge
            if (appElement.id === 'photos-app') {
                const badge = photosIcon.querySelector('.notification-badge');
                if (badge) badge.style.display = 'none';
            }
            
            // For phone app, remove notification badge
            if (appElement.id === 'phone-app') {
                const badge = phoneIcon.querySelector('.notification-badge');
                if (badge) badge.style.display = 'none';
            }
        }
        
        // Set up app icons click events
        notesIcon.addEventListener('click', () => openApp(notesApp));
        phoneIcon.addEventListener('click', () => openApp(phoneApp));
        messagesIcon.addEventListener('click', () => openApp(document.getElementById('messages-app')));
        mailIcon.addEventListener('click', () => openApp(document.getElementById('mail-app')));
        photosIcon.addEventListener('click', () => openApp(photosApp));
        musicIcon.addEventListener('click', () => openApp(document.getElementById('music-app')));
        weatherIcon.addEventListener('click', () => openApp(document.getElementById('weather-app')));
        calendarIcon.addEventListener('click', () => openApp(document.getElementById('calendar-app')));
        
        // Email functionality
        emailYaleResponse.addEventListener('click', () => {
            yaleResponseDetail.classList.add('active');
        });
        
        emailYaleRequest.addEventListener('click', () => {
            yaleRequestDetail.classList.add('active');
        });
        
        emailBack1.addEventListener('click', () => {
            yaleResponseDetail.classList.remove('active');
        });
        
        emailBack2.addEventListener('click', () => {
            yaleRequestDetail.classList.remove('active');
        });
        
        // Messages functionality
        unknownChat.addEventListener('click', () => {
            unknownChatDetail.classList.add('active');
        });
        
        chatBack.addEventListener('click', () => {
            unknownChatDetail.classList.remove('active');
        });
        
        // Populate messages with the provided conversation
        function populateMessages() {
            const messages = [
                {
                    date: 'March 2, 2024',
                    msgs: [
                        { text: 'Better stop while you can. You don\'t know who you\'re messing with.', type: 'received', time: '14:32' },
                        { text: 'Who are you? I don\'t have time for games.', type: 'sent', time: '14:45' }
                    ]
                },
                {
                    date: 'March 3, 2024',
                    msgs: [
                        { text: 'Your stubbornness will cost you dearly, Emma.', type: 'received', time: '09:17' },
                        { text: 'Tell me your name. I\'m warning you, I\'ll report this to the police.', type: 'sent', time: '09:25' },
                        { text: 'You really think the police can protect you?', type: 'received', time: '09:40' }
                    ]
                },
                {
                    date: 'March 5, 2024',
                    msgs: [
                        { text: 'Final warning. Don\'t mess with stories that don\'t belong to you.', type: 'received', time: '16:03' },
                        { text: 'I\'m not afraid of your threats. I\'m going to see this through to the end.', type: 'sent', time: '16:15' }
                    ]
                },
                {
                    date: 'March 7, 2024',
                    msgs: [
                        { text: 'You should have listened. It\'s too late now.', type: 'received', time: '19:28' },
                        { text: 'What does that mean?', type: 'sent', time: '19:35', read: true }
                    ]
                }
            ];
            
            unknownMessages.innerHTML = '';
            
            messages.forEach(dateGroup => {
                // Add date divider
                const dateDivider = document.createElement('div');
                dateDivider.className = 'date-divider';
                dateDivider.textContent = dateGroup.date;
                unknownMessages.appendChild(dateDivider);
                
                // Add messages for this date
                dateGroup.msgs.forEach(msg => {
                    const messageElement = document.createElement('div');
                    messageElement.className = `message ${msg.type}`;
                    
                    let messageHTML = `${msg.text}`;
                    messageHTML += `<div class="message-time">${msg.time}</div>`;
                    
                    messageElement.innerHTML = messageHTML;
                    unknownMessages.appendChild(messageElement);
                    
                    // Add read receipt for the last sent message
                    if (msg.type === 'sent' && msg.read) {
                        const readReceipt = document.createElement('div');
                        readReceipt.className = 'read-receipt';
                        readReceipt.textContent = 'Read';
                        unknownMessages.appendChild(readReceipt);
                    }
                });
            });
            
            // Scroll to the end of the conversation
            setTimeout(() => {
                unknownMessages.scrollTop = unknownMessages.scrollHeight;
            }, 100);
        }
        
        // Notes App Functionality
        
        // Update note count
        noteCount.textContent = `(${notes.length})`;

        // Populate notes list
        function populateNotesList() {
            notesList.innerHTML = '';
            notes.forEach(note => {
                const li = document.createElement('li');
                li.className = 'note-item';
                li.setAttribute('data-id', note.id);
                li.innerHTML = `
                    <div class="note-title">${note.title}</div>
                    <div class="note-preview">${note.preview}</div>
                    <div class="note-date">${note.date}</div>
                `;
                li.addEventListener('click', () => {
                    // Remove active class from all notes
                    document.querySelectorAll('.note-item').forEach(item => {
                        item.classList.remove('active');
                    });
                    
                    // Add active class to selected note
                    li.classList.add('active');
                    
                    // Display note content
                    displayNoteContent(note);
                    
                    // Show the note content
                    sidebar.style.display = 'none';
                    mainContent.classList.add('active');
                    document.querySelector('.note-title-display').textContent = note.title;
                });
                notesList.appendChild(li);
            });
        }

        // Display note content
        function displayNoteContent(note) {
            const formattedContent = note.content.replace(/\n/g, '</p><p>');
            noteContent.innerHTML = `<p>${formattedContent}</p>`;
            // Adjust empty paragraphs
            noteContent.innerHTML = noteContent.innerHTML.replace(/<p><\/p>/g, '<p>&nbsp;</p>');
        }

        // Back button functionality for notes
        backToList.addEventListener('click', () => {
            sidebar.style.display = 'block';
            mainContent.classList.remove('active');
        });
        
        // Gallery App Functionality

        // Populate gallery grid
        function populateGalleryGrid() {
            galleryGrid.innerHTML = '';
            gallery.forEach(item => {
                const galleryItem = document.createElement('div');
                galleryItem.className = 'gallery-item';
                galleryItem.style.backgroundImage = `url('${item.image}')`;
                galleryItem.setAttribute('data-id', item.id);
                
                galleryItem.addEventListener('click', () => {
                    showImageDetail(item);
                });
                
                galleryGrid.appendChild(galleryItem);
            });
        }

        // Show image detail
        function showImageDetail(item) {
            // Clone template
            const imageDetail = imageDetailTemplate.cloneNode(true);
            imageDetail.id = `image-detail-${item.id}`;
            imageDetail.classList.add('active');
            
            // Set image detail content
            const backButton = imageDetail.querySelector('.image-detail-back');
            const titleElement = imageDetail.querySelector('.image-detail-title');
            const imageElement = imageDetail.querySelector('.detail-image');
            const dateElement = imageDetail.querySelector('.image-detail-date');
            const descriptionElement = imageDetail.querySelector('.image-detail-description');
            
            titleElement.textContent = item.title;
            imageElement.src = item.image;
            imageElement.alt = item.title;
            dateElement.textContent = item.date;
            descriptionElement.textContent = item.description;
            
            // Back button functionality
            backButton.addEventListener('click', () => {
                imageDetail.classList.remove('active');
                setTimeout(() => {
                    imageDetail.remove();
                }, 300);
            });
            
            // Append to gallery app content
            document.querySelector('.gallery-app-content').appendChild(imageDetail);
        }
        
        // Phone App Functionality

        // Populate call log
        function populateCallLog() {
            callLog.innerHTML = '';
            phones.forEach(call => {
                const callItem = document.createElement('div');
                callItem.className = 'call-log-item';
                callItem.setAttribute('data-id', call.id);
                
                // Get initial for avatar
                const initial = call.name.charAt(0);
                
                // Determine call icon based on type
                let callIcon = '';
                let callTypeClass = '';
                
                if (call.type === 'outgoing') {
                    callIcon = '↗️';
                    callTypeClass = 'call-outgoing';
                } else if (call.type === 'incoming') {
                    callIcon = '↘️';
                    callTypeClass = 'call-incoming';
                } else if (call.type === 'missed') {
                    callIcon = '✕';
                    callTypeClass = 'call-missed';
                }
                
                callItem.innerHTML = `
                    <div class="call-avatar ${callTypeClass}">${initial}</div>
                    <div class="call-info-preview">
                        <div class="call-name">${call.name}</div>
                        <div class="call-number">${call.number} ${callIcon}</div>
                    </div>
                    <div class="call-time-preview">${call.date}</div>
                `;
                
                callItem.addEventListener('click', () => {
                    showCallDetail(call);
                });
                
                callLog.appendChild(callItem);
            });
        }

        // Show call detail
        function showCallDetail(call) {
            // Clone template
            const callDetail = callDetailTemplate.cloneNode(true);
            callDetail.id = `call-detail-${call.id}`;
            callDetail.classList.add('active');
            
            // Set call detail content
            const backButton = callDetail.querySelector('.call-detail-back');
            const avatarElement = callDetail.querySelector('.call-contact-avatar');
            const nameElement = callDetail.querySelector('.call-contact-name');
            const numberElement = callDetail.querySelector('.call-contact-number');
            const datetimeElement = callDetail.querySelector('.call-datetime');
            const durationElement = callDetail.querySelector('.call-duration');
            const typeElement = callDetail.querySelector('.call-type');
            const audioPlayer = callDetail.querySelector('.audio-player');
            const descriptionElement = callDetail.querySelector('.call-description');
            
            // Get initial for avatar
            const initial = call.name.charAt(0);
            avatarElement.textContent = initial;
            
            // Set call info
            nameElement.textContent = call.name;
            numberElement.textContent = call.number;
            datetimeElement.textContent = `Date: ${call.date} at ${call.time}`;
            durationElement.textContent = `Duration: ${call.duration}`;
            
            // Format call type
            let callTypeText = '';
            if (call.type === 'outgoing') {
                callTypeText = 'Outgoing call';
                avatarElement.classList.add('call-outgoing');
            } else if (call.type === 'incoming') {
                callTypeText = 'Incoming call';
                avatarElement.classList.add('call-incoming');
            } else if (call.type === 'missed') {
                callTypeText = 'Missed call';
                avatarElement.classList.add('call-missed');
            }
            
            typeElement.textContent = callTypeText;
            
            // Set audio source
            audioPlayer.src = call.audio;
            
            // Set description
            descriptionElement.textContent = call.description;
            
            // Back button functionality
            backButton.addEventListener('click', () => {
                // Pause audio if playing
                audioPlayer.pause();
                
                callDetail.classList.remove('active');
                setTimeout(() => {
                    callDetail.remove();
                }, 300);
            });
            
            // Append to phone app content
            document.querySelector('.phone-app-content').appendChild(callDetail);
        }

        // Initialize the app
        document.addEventListener('DOMContentLoaded', function() {
            populateNotesList();
            populateMessages();
            populateGalleryGrid();
            populateCallLog();
            
            // Check if email elements exist before adding the listeners
            if (emailYaleResponse && emailYaleRequest) {
                emailYaleResponse.addEventListener('click', () => {
                    yaleResponseDetail.classList.add('active');
                });
                
                emailYaleRequest.addEventListener('click', () => {
                    yaleRequestDetail.classList.add('active');
                });
            }
            
            // Check if chat elements exist
            if (unknownChat) {
                unknownChat.addEventListener('click', () => {
                    unknownChatDetail.classList.add('active');
                });
            }
        });

        // Function to simulate passcode key press to facilitate testing
        function simulatePasscodeEntry() {
            const correctCode = Array.from(correctPasscode);
            let index = 0;
            
            const enterInterval = setInterval(() => {
                if (index < 4) {
                    const key = document.querySelector(`.passcode-key[data-key="${correctCode[index]}"]`);
                    if (key) {
                        key.click();
                        index++;
                    }
                } else {
                    clearInterval(enterInterval);
                }
            }, 300);
        }

        // Function to facilitate unlocking during tests (uncomment to use)
        // document.addEventListener('keydown', function(e) {
        //     if (e.code === 'KeyU') {
        //         if (!homeScreen.classList.contains('active')) {
        //             simulatePasscodeEntry();
        //         }
        //     }
        // });

        // Support for different screen sizes
        function adjustForScreenSize() {
            const vh = window.innerHeight * 0.01;
            document.documentElement.style.setProperty('--vh', `${vh}px`);
            
            // Adjust dock position on small screens
            if (window.innerHeight < 600) {
                document.querySelector('.dock').style.bottom = '5px';
            } else {
                document.querySelector('.dock').style.bottom = '15px';
            }
        }

        // Adjust layout based on screen size
        window.addEventListener('resize', adjustForScreenSize);
        adjustForScreenSize();

        // Ensure virtual keyboard doesn't cause layout issues on mobile devices
        const inputs = document.querySelectorAll('input');
        inputs.forEach(input => {
            input.addEventListener('focus', () => {
                document.body.classList.add('keyboard-open');
            });
            
            input.addEventListener('blur', () => {
                document.body.classList.remove('keyboard-open');
            });
        });

        // Disable zoom on mobile devices for better experience
        document.addEventListener('gesturestart', function(e) {
            e.preventDefault();
        });

        // Function to reset the app to initial state (useful for testing)
        function resetApp() {
            // Close any open app
            if (currentApp) {
                currentApp.classList.remove('active');
                currentApp = null;
            }
            
            // Return to lock screen
            homeScreen.classList.remove('active');
            lockScreen.style.display = 'flex';
            passcodeScreen.classList.remove('active');
            resetPasscode();
            
            // Reset app details
            sidebar.style.display = 'block';
            mainContent.classList.remove('active');
            yaleResponseDetail.classList.remove('active');
            yaleRequestDetail.classList.remove('active');
            unknownChatDetail.classList.remove('active');
            
            // Close any open image or call details
            document.querySelectorAll('.image-detail.active').forEach(elem => {
                elem.classList.remove('active');
                setTimeout(() => elem.remove(), 300);
            });
            
            document.querySelectorAll('.call-detail.active').forEach(elem => {
                // Pause audio if playing
                const audio = elem.querySelector('audio');
                if (audio) audio.pause();
                
                elem.classList.remove('active');
                setTimeout(() => elem.remove(), 300);
            });
            
            // Restore notifications
            const messageBadge = messagesIcon.querySelector('.notification-badge');
            const mailBadge = mailIcon.querySelector('.notification-badge');
            const photosBadge = photosIcon.querySelector('.notification-badge');
            const phoneBadge = phoneIcon.querySelector('.notification-badge');
            
            if (messageBadge) messageBadge.style.display = 'flex';
            if (mailBadge) mailBadge.style.display = 'flex';
            if (photosBadge) photosBadge.style.display = 'flex';
            if (phoneBadge) phoneBadge.style.display = 'flex';
        }

        // For production use, remove the console.log
        console.log = function() {};
    </script>
</body>
</html>