<?php
// Start session for authentication
session_start();

// Admin settings
$admin_username = "admin";
$admin_password = "detective123"; // Recommended to use hash in production

// Check login
$is_logged_in = isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;

// Process login
if (isset($_POST['login'])) {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if ($username === $admin_username && $password === $admin_password) {
        $_SESSION['admin_logged_in'] = true;
        $is_logged_in = true;
        // Redirect to clear POST data
        header("Location: admin.php");
        exit;
    } else {
        $login_error = "Invalid credentials";
    }
}

// Process logout
if (isset($_GET['logout'])) {
    session_destroy();
    header("Location: admin.php");
    exit;
}

// Define paths for data files
$data_dir = 'data';
$notesJsonPath = $data_dir . '/notes.json';
$galleryJsonPath = $data_dir . '/gallery.json';
$phonesJsonPath = $data_dir . '/phones.json';
$messagesJsonPath = $data_dir . '/messages.json';

// Create directory if it doesn't exist
if (!is_dir($data_dir)) {
    mkdir($data_dir, 0755, true);
}
if (!is_dir($data_dir . '/images')) {
    mkdir($data_dir . '/images', 0755, true);
}
if (!is_dir($data_dir . '/audio')) {
    mkdir($data_dir . '/audio', 0755, true);
}

// Utility Functions
function loadJSON($file_path, $default = []) {
    if (file_exists($file_path)) {
        $json = file_get_contents($file_path);
        return json_decode($json, true) ?: $default;
    }
    return $default;
}

function saveJSON($file_path, $data) {
    file_put_contents($file_path, json_encode($data, JSON_PRETTY_PRINT));
}

function getNextId($data) {
    if (empty($data)) {
        return 1;
    }
    return max(array_column($data, 'id')) + 1;
}

function uploadFile($file, $target_dir) {
    // Check if there is a file to upload
    if (!isset($file['tmp_name']) || empty($file['tmp_name'])) {
        return null;
    }
    
    $target_file = $target_dir . '/' . basename($file["name"]);
    $upload_ok = 1;
    $file_type = strtolower(pathinfo($target_file, PATHINFO_EXTENSION));
    
    // Check file size
    if ($file["size"] > 10000000) { // 10MB
        return ["error" => "File too large."];
        $upload_ok = 0;
    }
    
    // Allow certain file formats
    if ($target_dir == 'data/images') {
        $allowed_types = ["jpg", "jpeg", "png", "gif"];
        if (!in_array($file_type, $allowed_types)) {
            return ["error" => "Only JPG, JPEG, PNG and GIF are allowed."];
            $upload_ok = 0;
        }
    } elseif ($target_dir == 'data/audio') {
        $allowed_types = ["mp3", "wav", "ogg"];
        if (!in_array($file_type, $allowed_types)) {
            return ["error" => "Only MP3, WAV and OGG are allowed."];
            $upload_ok = 0;
        }
    }
    
    // Rename file to avoid overwriting
    $new_filename = uniqid() . '.' . $file_type;
    $target_file = $target_dir . '/' . $new_filename;
    
    // Upload the file
    if ($upload_ok == 1) {
        if (move_uploaded_file($file["tmp_name"], $target_file)) {
            return ["success" => true, "file_path" => $target_file];
        } else {
            return ["error" => "An error occurred while uploading the file."];
        }
    }
    
    return ["error" => "An unknown error occurred."];
}

// Notes Handling
function handleNotes() {
    global $notesJsonPath;
    $notes = loadJSON($notesJsonPath);
    $message = '';
    
    // Add or update note
    if (isset($_POST['save_note'])) {
        $id = $_POST['note_id'] ?? 0;
        $title = $_POST['note_title'] ?? '';
        $content = $_POST['note_content'] ?? '';
        $date = $_POST['note_date'] ?? date('d/m/Y');
        $preview = substr(strip_tags($content), 0, 100) . '...';
        
        $note = [
            'title' => $title,
            'content' => $content,
            'date' => $date,
            'preview' => $preview
        ];
        
        if ($id == 0) {
            // New note
            $note['id'] = getNextId($notes);
            $notes[] = $note;
            $message = 'Note added successfully!';
        } else {
            // Update existing note
            foreach ($notes as $key => $existing_note) {
                if ($existing_note['id'] == $id) {
                    $note['id'] = $id;
                    $notes[$key] = $note;
                    break;
                }
            }
            $message = 'Note updated successfully!';
        }
        
        saveJSON($notesJsonPath, $notes);
    }
    
    // Delete note
    if (isset($_GET['delete_note'])) {
        $id = intval($_GET['delete_note']);
        
        foreach ($notes as $key => $note) {
            if ($note['id'] == $id) {
                unset($notes[$key]);
                break;
            }
        }
        
        $notes = array_values($notes); // Reorganize indexes
        saveJSON($notesJsonPath, $notes);
        $message = 'Note deleted successfully!';
    }
    
    return [
        'notes' => $notes,
        'message' => $message
    ];
}

// Gallery Handling
function handleGallery() {
    global $galleryJsonPath;
    $gallery = loadJSON($galleryJsonPath);
    $message = '';
    $error = '';
    
    // Add or update photo
    if (isset($_POST['save_photo'])) {
        $id = $_POST['photo_id'] ?? 0;
        $title = $_POST['photo_title'] ?? '';
        $description = $_POST['photo_description'] ?? '';
        $date = $_POST['photo_date'] ?? date('d/m/Y');
        
        $photo = [
            'title' => $title,
            'description' => $description,
            'date' => $date
        ];
        
        // Check if there's an uploaded file
        if (!empty($_FILES['photo_image']['tmp_name'])) {
            $upload_result = uploadFile($_FILES['photo_image'], 'data/images');
            
            if (isset($upload_result['success'])) {
                $photo['image'] = $upload_result['file_path'];
            } else if (isset($upload_result['error'])) {
                $error = $upload_result['error'];
            }
        } else if ($id > 0) {
            // Keep existing image if editing
            foreach ($gallery as $existing_photo) {
                if ($existing_photo['id'] == $id) {
                    $photo['image'] = $existing_photo['image'];
                    break;
                }
            }
        } else {
            $error = 'Select an image to upload.';
        }
        
        if (empty($error)) {
            if ($id == 0) {
                // New photo
                $photo['id'] = getNextId($gallery);
                $gallery[] = $photo;
                $message = 'Photo added successfully!';
            } else {
                // Update existing photo
                foreach ($gallery as $key => $existing_photo) {
                    if ($existing_photo['id'] == $id) {
                        $photo['id'] = $id;
                        $gallery[$key] = $photo;
                        break;
                    }
                }
                $message = 'Photo updated successfully!';
            }
            
            saveJSON($galleryJsonPath, $gallery);
        }
    }
    
    // Delete photo
    if (isset($_GET['delete_photo'])) {
        $id = intval($_GET['delete_photo']);
        
        foreach ($gallery as $key => $photo) {
            if ($photo['id'] == $id) {
                // Remove the physical file
                if (file_exists($photo['image'])) {
                    unlink($photo['image']);
                }
                
                unset($gallery[$key]);
                break;
            }
        }
        
        $gallery = array_values($gallery); // Reorganize indexes
        saveJSON($galleryJsonPath, $gallery);
        $message = 'Photo deleted successfully!';
    }
    
    return [
        'gallery' => $gallery,
        'message' => $message,
        'error' => $error
    ];
}

// Phone Calls Handling
function handlePhones() {
    global $phonesJsonPath;
    $phones = loadJSON($phonesJsonPath);
    $message = '';
    $error = '';
    
    // Add or update call
    if (isset($_POST['save_call'])) {
        $id = $_POST['call_id'] ?? 0;
        $name = $_POST['call_name'] ?? '';
        $number = $_POST['call_number'] ?? '';
        $date = $_POST['call_date'] ?? date('d/m/Y');
        $time = $_POST['call_time'] ?? date('H:i');
        $duration = $_POST['call_duration'] ?? '0:00';
        $type = $_POST['call_type'] ?? 'incoming';
        $description = $_POST['call_description'] ?? '';
        
        $call = [
            'name' => $name,
            'number' => $number,
            'date' => $date,
            'time' => $time,
            'duration' => $duration,
            'type' => $type,
            'description' => $description
        ];
        
        // Check if there's an uploaded file
        if (!empty($_FILES['call_audio']['tmp_name'])) {
            $upload_result = uploadFile($_FILES['call_audio'], 'data/audio');
            
            if (isset($upload_result['success'])) {
                $call['audio'] = $upload_result['file_path'];
            } else if (isset($upload_result['error'])) {
                $error = $upload_result['error'];
            }
        } else if ($id > 0) {
            // Keep existing audio if editing
            foreach ($phones as $existing_call) {
                if ($existing_call['id'] == $id) {
                    $call['audio'] = $existing_call['audio'];
                    break;
                }
            }
        } else {
            $error = 'Select an audio file to upload.';
        }
        
        if (empty($error)) {
            if ($id == 0) {
                // New call
                $call['id'] = getNextId($phones);
                $phones[] = $call;
                $message = 'Call added successfully!';
            } else {
                // Update existing call
                foreach ($phones as $key => $existing_call) {
                    if ($existing_call['id'] == $id) {
                        $call['id'] = $id;
                        $phones[$key] = $call;
                        break;
                    }
                }
                $message = 'Call updated successfully!';
            }
            
            saveJSON($phonesJsonPath, $phones);
        }
    }
    
    // Delete call
    if (isset($_GET['delete_call'])) {
        $id = intval($_GET['delete_call']);
        
        foreach ($phones as $key => $call) {
            if ($call['id'] == $id) {
                // Remove the physical file
                if (file_exists($call['audio'])) {
                    unlink($call['audio']);
                }
                
                unset($phones[$key]);
                break;
            }
        }
        
        $phones = array_values($phones); // Reorganize indexes
        saveJSON($phonesJsonPath, $phones);
        $message = 'Call deleted successfully!';
    }
    
    return [
        'phones' => $phones,
        'message' => $message,
        'error' => $error
    ];
}

// Messages Handling
function handleMessages() {
    global $messagesJsonPath;
    $messages = loadJSON($messagesJsonPath, [
        'unknown' => [
            'name' => 'Unknown',
            'conversations' => [
                [
                    'date' => 'March 02, 2024',
                    'msgs' => [
                        [ 'text' => 'Better stop while you can. You don\'t know who you\'re messing with.', 'type' => 'received', 'time' => '14:32' ],
                        [ 'text' => 'Who are you? I don\'t have time for games.', 'type' => 'sent', 'time' => '14:45' ]
                    ]
                ],
                [
                    'date' => 'March 03, 2024',
                    'msgs' => [
                        [ 'text' => 'Your stubbornness will cost you dearly, Emma.', 'type' => 'received', 'time' => '09:17' ],
                        [ 'text' => 'Tell me your name. I\'m warning you, I\'ll report this to the police.', 'type' => 'sent', 'time' => '09:25' ],
                        [ 'text' => 'Do you really think the police can protect you?', 'type' => 'received', 'time' => '09:40' ]
                    ]
                ],
                [
                    'date' => 'March 05, 2024',
                    'msgs' => [
                        [ 'text' => 'Final warning. Don\'t meddle in stories that don\'t belong to you.', 'type' => 'received', 'time' => '16:03' ],
                        [ 'text' => 'I\'m not afraid of your threats. I\'ll take this to the end.', 'type' => 'sent', 'time' => '16:15' ]
                    ]
                ],
                [
                    'date' => 'March 07, 2024',
                    'msgs' => [
                        [ 'text' => 'You should have listened. Now it\'s too late.', 'type' => 'received', 'time' => '19:28' ],
                        [ 'text' => 'What does that mean?', 'type' => 'sent', 'time' => '19:35', 'read' => true ]
                    ]
                ]
            ]
        ]
    ]);
    $message = '';
    
    // Add or update message
    if (isset($_POST['save_message'])) {
        $contact_id = $_POST['contact_id'] ?? 'unknown';
        $date = $_POST['message_date'] ?? date('F d, Y');
        $text = $_POST['message_text'] ?? '';
        $time = $_POST['message_time'] ?? date('H:i');
        $type = $_POST['message_type'] ?? 'received';
        $read = isset($_POST['message_read']) ? true : false;
        
        $new_message = [
            'text' => $text,
            'type' => $type,
            'time' => $time
        ];
        
        if ($read) {
            $new_message['read'] = true;
        }
        
        // Check if contact already exists
        if (!isset($messages[$contact_id])) {
            $messages[$contact_id] = [
                'name' => $_POST['contact_name'] ?? 'Contact',
                'conversations' => []
            ];
        }
        
        // Check if date already exists
        $date_exists = false;
        foreach ($messages[$contact_id]['conversations'] as $key => $conversation) {
            if ($conversation['date'] == $date) {
                $messages[$contact_id]['conversations'][$key]['msgs'][] = $new_message;
                $date_exists = true;
                break;
            }
        }
        
        // If date doesn't exist, create a new entry
        if (!$date_exists) {
            $messages[$contact_id]['conversations'][] = [
                'date' => $date,
                'msgs' => [$new_message]
            ];
        }
        
        saveJSON($messagesJsonPath, $messages);
        $message = 'Message added successfully!';
    }
    
    // Remove message group
    if (isset($_GET['delete_message_group'])) {
        $parts = explode(':', $_GET['delete_message_group']);
        
        if (count($parts) == 2) {
            $contact_id = $parts[0];
            $date_index = intval($parts[1]);
            
            if (isset($messages[$contact_id]) && isset($messages[$contact_id]['conversations'][$date_index])) {
                array_splice($messages[$contact_id]['conversations'], $date_index, 1);
                
                // If there are no more conversations for this contact, remove the contact
                if (empty($messages[$contact_id]['conversations'])) {
                    unset($messages[$contact_id]);
                }
                
                saveJSON($messagesJsonPath, $messages);
                $message = 'Message group removed successfully!';
            }
        }
    }
    
    // Remove individual message
    if (isset($_GET['delete_message'])) {
        $parts = explode(':', $_GET['delete_message']);
        
        if (count($parts) == 3) {
            $contact_id = $parts[0];
            $date_index = intval($parts[1]);
            $message_index = intval($parts[2]);
            
            if (isset($messages[$contact_id]) && 
                isset($messages[$contact_id]['conversations'][$date_index]) && 
                isset($messages[$contact_id]['conversations'][$date_index]['msgs'][$message_index])) {
                
                array_splice($messages[$contact_id]['conversations'][$date_index]['msgs'], $message_index, 1);
                
                // If there are no more messages for this date, remove the date
                if (empty($messages[$contact_id]['conversations'][$date_index]['msgs'])) {
                    array_splice($messages[$contact_id]['conversations'], $date_index, 1);
                    
                    // If there are no more conversations for this contact, remove the contact
                    if (empty($messages[$contact_id]['conversations'])) {
                        unset($messages[$contact_id]);
                    }
                }
                
                saveJSON($messagesJsonPath, $messages);
                $message = 'Message removed successfully!';
            }
        }
    }
    
    return [
        'messages' => $messages,
        'message' => $message
    ];
}

// Load data based on active tab
$active_tab = $_GET['tab'] ?? 'dashboard';
$notes_data = [];
$gallery_data = [];
$phones_data = [];
$messages_data = [];

if ($is_logged_in) {
    switch ($active_tab) {
        case 'notes':
            $notes_data = handleNotes();
            break;
            
        case 'gallery':
            $gallery_data = handleGallery();
            break;
            
        case 'phones':
            $phones_data = handlePhones();
            break;
            
        case 'messages':
            $messages_data = handleMessages();
            break;
            
        default:
            // Load counts for dashboard
            $notes_data['notes'] = loadJSON($notesJsonPath);
            $gallery_data['gallery'] = loadJSON($galleryJsonPath);
            $phones_data['phones'] = loadJSON($phonesJsonPath);
            $messages_data['messages'] = loadJSON($messagesJsonPath);
            break;
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel de Administração - iPhone Emulator</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        :root {
            --primary-color: #2c3e50;
            --secondary-color: #34495e;
            --accent-color: #3498db;
            --light-color: #ecf0f1;
            --dark-color: #2c3e50;
            --success-color: #2ecc71;
            --warning-color: #f39c12;
            --danger-color: #e74c3c;
        }
        
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .login-container {
            max-width: 400px;
            margin: 100px auto;
            padding: 30px;
            background-color: #fff;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .sidebar {
            background-color: var(--primary-color);
            color: white;
            min-height: 100vh;
            padding-top: 20px;
        }
        
        .sidebar-header {
            padding: 15px;
            text-align: center;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            border-radius: 5px;
            margin: 5px 10px;
            transition: all 0.3s;
        }
        
        .nav-link:hover, .nav-link.active {
            background-color: var(--accent-color);
            color: white;
        }
        
        .nav-link i {
            margin-right: 10px;
        }
        
        .main-content {
            padding: 30px;
        }
        
        .dashboard-card {
            background-color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            transition: transform 0.3s;
        }
        
        .dashboard-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .card-icon {
            font-size: 2.5rem;
            margin-bottom: 15px;
            color: var(--accent-color);
        }
        
        .form-control, .form-select {
            border: 1px solid #ced4da;
            border-radius: 5px;
            padding: 10px 15px;
        }
        
        .btn-primary {
            background-color: var(--accent-color);
            border-color: var(--accent-color);
        }
        
        .btn-primary:hover {
            background-color: #2980b9;
            border-color: #2980b9;
        }
        
        .table {
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            border-radius: 10px;
            overflow: hidden;
        }
        
        .table th {
            background-color: var(--primary-color);
            color: white;
        }
        
        .action-buttons .btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        
        .gallery-preview {
            width: 100px;
            height: 60px;
            object-fit: cover;
            border-radius: 5px;
        }
        
        .phone-preview {
            width: 30px;
            height: 30px;
            display: inline-flex;
            justify-content: center;
            align-items: center;
            border-radius: 50%;
            color: white;
            font-weight: bold;
        }
        
        .phone-incoming {
            background-color: #3498db;
        }
        
        .phone-outgoing {
            background-color: #2ecc71;
        }
        
        .phone-missed {
            background-color: #e74c3c;
        }
        
        .message-container {
            background-color: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .message-date {
            background-color: var(--primary-color);
            color: white;
            padding: 5px 10px;
            border-radius: 5px;
            display: inline-block;
            margin-bottom: 10px;
        }
        
        .message-item {
            padding: 10px;
            border-radius: 10px;
            margin-bottom: 10px;
            position: relative;
        }
        
        .message-received {
            background-color: #e9ecef;
            margin-right: 20%;
        }
        
        .message-sent {
            background-color: #d4edff;
            margin-left: 20%;
            text-align: right;
        }
        
        .message-time {
            font-size: 0.75rem;
            color: #6c757d;
            margin-top: 5px;
        }
        
        .delete-message {
            position: absolute;
            top: 5px;
            right: 5px;
            color: #e74c3c;
            cursor: pointer;
            display: none;
        }
        
        .message-item:hover .delete-message {
            display: block;
        }
        
        .contact-header {
            background-color: var(--primary-color);
            color: white;
            padding: 10px 15px;
            border-radius: 10px 10px 0 0;
            margin-bottom: 0;
        }
        
        /* Alternar previews em dispositivos menores */
        @media (max-width: 768px) {
            .table-responsive {
                overflow-x: auto;
            }
            
            .gallery-preview {
                width: 60px;
                height: 40px;
            }
        }
        
        /* Barra de rolagem personalizada */
        ::-webkit-scrollbar {
            width: 8px;
            height: 8px;
        }
        
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
        }
        
        ::-webkit-scrollbar-thumb {
            background: var(--accent-color);
            border-radius: 10px;
        }
        
        ::-webkit-scrollbar-thumb:hover {
            background: #2980b9;
        }
    </style>
</head>
<body>
    <?php if (!$is_logged_in): ?>
    <!-- Tela de Login -->
    <div class="login-container">
        <h2 class="text-center mb-4">Painel de Administração</h2>
        
        <?php if (isset($login_error)): ?>
            <div class="alert alert-danger" role="alert">
                <?php echo $login_error; ?>
            </div>
        <?php endif; ?>
        
        <form method="post">
            <div class="mb-3">
                <label for="username" class="form-label">Usuário</label>
                <input type="text" class="form-control" id="username" name="username" required>
            </div>
            <div class="mb-3">
                <label for="password" class="form-label">Senha</label>
                <input type="password" class="form-control" id="password" name="password" required>
            </div>
            <button type="submit" name="login" class="btn btn-primary w-100">Entrar</button>
        </form>
    </div>
    <?php else: ?>
    <!-- Painel de Administração -->
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar">
                <div class="sidebar-header">
                    <h4>iPhone Admin</h4>
                </div>
                <ul class="nav flex-column">
                    <li class="nav-item">
                        <a class="nav-link <?php echo $active_tab == 'dashboard' ? 'active' : ''; ?>" href="?tab=dashboard">
                            <i class="bi bi-speedometer2"></i> Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $active_tab == 'notes' ? 'active' : ''; ?>" href="?tab=notes">
                            <i class="bi bi-journal-text"></i> Notes
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $active_tab == 'gallery' ? 'active' : ''; ?>" href="?tab=gallery">
                            <i class="bi bi-image"></i> Gallery
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $active_tab == 'phones' ? 'active' : ''; ?>" href="?tab=phones">
                            <i class="bi bi-telephone"></i> Calls
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $active_tab == 'messages' ? 'active' : ''; ?>" href="?tab=messages">
                            <i class="bi bi-chat-dots"></i> Messages
                        </a>
                    </li>
                    <li class="nav-item mt-5">
                        <a class="nav-link text-danger" href="?logout=1">
                            <i class="bi bi-box-arrow-right"></i> Logout
                        </a>
                    </li>
                </ul>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 main-content">
                <?php if ($active_tab == 'dashboard'): ?>
                <!-- Dashboard -->
                <h2 class="mb-4">Dashboard</h2>
                
                <div class="row">
                    <div class="col-md-6 col-lg-3">
                        <div class="dashboard-card">
                            <div class="card-icon">
                                <i class="bi bi-journal-text"></i>
                            </div>
                            <h5>Notes</h5>
                            <p class="fs-4"><?php echo count($notes_data['notes']); ?> items</p>
                            <a href="?tab=notes" class="btn btn-sm btn-outline-primary">Manage</a>
                        </div>
                    </div>
                    
                    <div class="col-md-6 col-lg-3">
                        <div class="dashboard-card">
                            <div class="card-icon">
                                <i class="bi bi-image"></i>
                            </div>
                            <h5>Gallery</h5>
                            <p class="fs-4"><?php echo count($gallery_data['gallery']); ?> items</p>
                            <a href="?tab=gallery" class="btn btn-sm btn-outline-primary">Manage</a>
                        </div>
                    </div>
                    
                    <div class="col-md-6 col-lg-3">
                        <div class="dashboard-card">
                            <div class="card-icon">
                                <i class="bi bi-telephone"></i>
                            </div>
                            <h5>Calls</h5>
                            <p class="fs-4"><?php echo count($phones_data['phones']); ?> items</p>
                            <a href="?tab=phones" class="btn btn-sm btn-outline-primary">Manage</a>
                        </div>
                    </div>
                    
                    <div class="col-md-6 col-lg-3">
                        <div class="dashboard-card">
                            <div class="card-icon">
                                <i class="bi bi-chat-dots"></i>
                            </div>
                            <h5>Messages</h5>
                            <p class="fs-4"><?php echo count($messages_data['messages']); ?> contacts</p>
                            <a href="?tab=messages" class="btn btn-sm btn-outline-primary">Manage</a>
                        </div>
                    </div>
                </div>
                
                <div class="row mt-4">
                    <div class="col-12">
                        <div class="dashboard-card">
                            <h4>Welcome to the Admin Panel</h4>
                            <p>This panel allows you to manage all the content of your iPhone emulator.</p>
                            <p>Use the sidebar to access different sections:</p>
                            <ul>
                                <li><strong>Notes:</strong> Manage notes and diaries found during the investigation.</li>
                                <li><strong>Gallery:</strong> Add and manage visual evidence photos.</li>
                                <li><strong>Calls:</strong> Configure call history and audio conversations.</li>
                                <li><strong>Messages:</strong> Create conversations and message exchanges between characters.</li>
                            </ul>
                            <p>The iPhone emulator will automatically load content when the user accesses the app.</p>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($active_tab == 'notes'): ?>
                <!-- Notes -->
                <h2 class="mb-4">Manage Notes</h2>
                
                <?php if (!empty($notes_data['message'])): ?>
                    <div class="alert alert-success" role="alert">
                        <?php echo $notes_data['message']; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Note Form -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Add / Edit Note</h5>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <input type="hidden" id="note_id" name="note_id" value="0">
                            
                            <div class="row mb-3">
                                <div class="col-md-8">
                                    <label for="note_title" class="form-label">Title</label>
                                    <input type="text" class="form-control" id="note_title" name="note_title" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="note_date" class="form-label">Date</label>
                                    <input type="text" class="form-control" id="note_date" name="note_date" value="<?php echo date('d/m/Y'); ?>" required>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="note_content" class="form-label">Content</label>
                                <textarea class="form-control" id="note_content" name="note_content" rows="6" required></textarea>
                            </div>
                            
                            <div class="d-flex justify-content-between">
                                <button type="button" class="btn btn-secondary" onclick="resetNoteForm()">Clear</button>
                                <button type="submit" name="save_note" class="btn btn-primary">Save Note</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Notes List -->
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Existing Notes</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Title</th>
                                        <th>Date</th>
                                        <th>Preview</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($notes_data['notes'] as $note): ?>
                                    <tr>
                                        <td><?php echo $note['id']; ?></td>
                                        <td><?php echo htmlspecialchars($note['title']); ?></td>
                                        <td><?php echo $note['date']; ?></td>
                                        <td><?php echo htmlspecialchars(substr($note['preview'], 0, 50) . '...'); ?></td>
                                        <td class="action-buttons">
                                            <button class="btn btn-sm btn-info" onclick="editNote(<?php echo htmlspecialchars(json_encode($note)); ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <a href="?tab=notes&delete_note=<?php echo $note['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this note?')">
                                                <i class="bi bi-trash"></i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    
                                    <?php if (empty($notes_data['notes'])): ?>
                                    <tr>
                                        <td colspan="5" class="text-center">No notes found.</td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($active_tab == 'gallery'): ?>
                <!-- Gallery -->
                <h2 class="mb-4">Manage Gallery</h2>
                
                <?php if (!empty($gallery_data['message'])): ?>
                    <div class="alert alert-success" role="alert">
                        <?php echo $gallery_data['message']; ?>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($gallery_data['error'])): ?>
                    <div class="alert alert-danger" role="alert">
                        <?php echo $gallery_data['error']; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Photo Form -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Add / Edit Photo</h5>
                    </div>
                    <div class="card-body">
                        <form method="post" enctype="multipart/form-data">
                            <input type="hidden" id="photo_id" name="photo_id" value="0">
                            
                            <div class="row mb-3">
                                <div class="col-md-8">
                                    <label for="photo_title" class="form-label">Title</label>
                                    <input type="text" class="form-control" id="photo_title" name="photo_title" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="photo_date" class="form-label">Date</label>
                                    <input type="text" class="form-control" id="photo_date" name="photo_date" value="<?php echo date('d/m/Y'); ?>" required>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="photo_description" class="form-label">Description</label>
                                <textarea class="form-control" id="photo_description" name="photo_description" rows="3" required></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label for="photo_image" class="form-label">Image</label>
                                <input type="file" class="form-control" id="photo_image" name="photo_image" accept="image/*">
                                <div id="photo_preview_container" class="mt-2 d-none">
                                    <p>Current image:</p>
                                    <img id="photo_preview" src="" alt="Preview" style="max-width: 200px; max-height: 150px; object-fit: contain;">
                                </div>
                            </div>
                            
                            <div class="d-flex justify-content-between">
                                <button type="button" class="btn btn-secondary" onclick="resetPhotoForm()">Clear</button>
                                <button type="submit" name="save_photo" class="btn btn-primary">Save Photo</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Photos List -->
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Existing Photos</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Image</th>
                                        <th>Title</th>
                                        <th>Date</th>
                                        <th>Description</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($gallery_data['gallery'] as $photo): ?>
                                    <tr>
                                        <td><?php echo $photo['id']; ?></td>
                                        <td>
                                            <img src="<?php echo $photo['image']; ?>" alt="<?php echo htmlspecialchars($photo['title']); ?>" class="gallery-preview">
                                        </td>
                                        <td><?php echo htmlspecialchars($photo['title']); ?></td>
                                        <td><?php echo $photo['date']; ?></td>
                                        <td><?php echo htmlspecialchars(substr($photo['description'], 0, 50) . '...'); ?></td>
                                        <td class="action-buttons">
                                            <button class="btn btn-sm btn-info" onclick="editPhoto(<?php echo htmlspecialchars(json_encode($photo)); ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <a href="?tab=gallery&delete_photo=<?php echo $photo['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this photo?')">
                                                <i class="bi bi-trash"></i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    
                                    <?php if (empty($gallery_data['gallery'])): ?>
                                    <tr>
                                        <td colspan="6" class="text-center">No photos found.</td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($active_tab == 'phones'): ?>
                <!-- Calls -->
                <h2 class="mb-4">Manage Calls</h2>
                
                <?php if (!empty($phones_data['message'])): ?>
                    <div class="alert alert-success" role="alert">
                        <?php echo $phones_data['message']; ?>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($phones_data['error'])): ?>
                    <div class="alert alert-danger" role="alert">
                        <?php echo $phones_data['error']; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Call Form -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Add / Edit Call</h5>
                    </div>
                    <div class="card-body">
                        <form method="post" enctype="multipart/form-data">
                            <input type="hidden" id="call_id" name="call_id" value="0">
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="call_name" class="form-label">Contact Name</label>
                                    <input type="text" class="form-control" id="call_name" name="call_name" required>
                                </div>
                                <div class="col-md-6">
                                    <label for="call_number" class="form-label">Phone Number</label>
                                    <input type="text" class="form-control" id="call_number" name="call_number" required>
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="call_date" class="form-label">Date</label>
                                    <input type="text" class="form-control" id="call_date" name="call_date" value="<?php echo date('d/m/Y'); ?>" required>
                                </div>
                                <div class="col-md-3">
                                    <label for="call_time" class="form-label">Time</label>
                                    <input type="text" class="form-control" id="call_time" name="call_time" value="<?php echo date('H:i'); ?>" required>
                                </div>
                                <div class="col-md-2">
                                    <label for="call_duration" class="form-label">Duration</label>
                                    <input type="text" class="form-control" id="call_duration" name="call_duration" value="0:00" required>
                                </div>
                                <div class="col-md-3">
                                    <label for="call_type" class="form-label">Type</label>
                                    <select class="form-select" id="call_type" name="call_type" required>
                                        <option value="incoming">Received</option>
                                        <option value="outgoing">Made</option>
                                        <option value="missed">Missed</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="call_description" class="form-label">Description</label>
                                <textarea class="form-control" id="call_description" name="call_description" rows="3" required></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label for="call_audio" class="form-label">Audio File</label>
                                <input type="file" class="form-control" id="call_audio" name="call_audio" accept="audio/*">
                                <div id="call_audio_preview_container" class="mt-2 d-none">
                                    <p>Current audio:</p>
                                    <audio id="call_audio_preview" controls>
                                        <source src="" type="audio/mpeg">
                                        Your browser does not support the audio element.
                                    </audio>
                                </div>
                            </div>
                            
                            <div class="d-flex justify-content-between">
                                <button type="button" class="btn btn-secondary" onclick="resetCallForm()">Clear</button>
                                <button type="submit" name="save_call" class="btn btn-primary">Save Call</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Calls List -->
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Existing Calls</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Contact</th>
                                        <th>Number</th>
                                        <th>Date/Time</th>
                                        <th>Type</th>
                                        <th>Duration</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($phones_data['phones'] as $call): ?>
                                    <tr>
                                        <td><?php echo $call['id']; ?></td>
                                        <td>
                                            <?php
                                            $type_class = '';
                                            $initial = substr($call['name'], 0, 1);
                                            
                                            if ($call['type'] == 'incoming') {
                                                $type_class = 'phone-incoming';
                                            } elseif ($call['type'] == 'outgoing') {
                                                $type_class = 'phone-outgoing';
                                            } elseif ($call['type'] == 'missed') {
                                                $type_class = 'phone-missed';
                                            }
                                            ?>
                                            <div class="phone-preview <?php echo $type_class; ?>">
                                                <?php echo $initial; ?>
                                            </div>
                                            <?php echo htmlspecialchars($call['name']); ?>
                                        </td>
                                        <td><?php echo $call['number']; ?></td>
                                        <td><?php echo $call['date']; ?> - <?php echo $call['time']; ?></td>
                                        <td>
                                            <?php
                                            if ($call['type'] == 'incoming') {
                                                echo '<span class="badge bg-primary">Received</span>';
                                            } elseif ($call['type'] == 'outgoing') {
                                                echo '<span class="badge bg-success">Made</span>';
                                            } elseif ($call['type'] == 'missed') {
                                                echo '<span class="badge bg-danger">Missed</span>';
                                            }
                                            ?>
                                        </td>
                                        <td><?php echo $call['duration']; ?></td>
                                        <td class="action-buttons">
                                            <button class="btn btn-sm btn-info" onclick="editCall(<?php echo htmlspecialchars(json_encode($call)); ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <a href="?tab=phones&delete_call=<?php echo $call['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this call?')">
                                                <i class="bi bi-trash"></i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    
                                    <?php if (empty($phones_data['phones'])): ?>
                                    <tr>
                                        <td colspan="7" class="text-center">No calls found.</td>
                                    </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($active_tab == 'messages'): ?>
                <!-- Messages -->
                <h2 class="mb-4">Manage Messages</h2>
                
                <?php if (!empty($messages_data['message'])): ?>
                    <div class="alert alert-success" role="alert">
                        <?php echo $messages_data['message']; ?>
                    </div>
                <?php endif; ?>
                
                <!-- Message Form -->
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Add Message</h5>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="contact_id" class="form-label">Contact</label>
                                    <select class="form-select" id="contact_id" name="contact_id">
                                        <option value="unknown">Unknown</option>
                                        <?php 
                                        foreach ($messages_data['messages'] as $contact_id => $contact) {
                                            if ($contact_id != 'unknown') {
                                                echo '<option value="' . $contact_id . '">' . htmlspecialchars($contact['name']) . '</option>';
                                            }
                                        }
                                        ?>
                                        <option value="new">New Contact...</option>
                                    </select>
                                </div>
                                <div class="col-md-6 new-contact-field d-none">
                                    <label for="contact_name" class="form-label">Name of New Contact</label>
                                    <input type="text" class="form-control" id="contact_name" name="contact_name">
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="message_date" class="form-label">Date</label>
                                    <input type="text" class="form-control" id="message_date" name="message_date" value="<?php echo date('F d, Y'); ?>" required>
                                </div>
                                <div class="col-md-3">
                                    <label for="message_time" class="form-label">Time</label>
                                    <input type="text" class="form-control" id="message_time" name="message_time" value="<?php echo date('H:i'); ?>" required>
                                </div>
                                <div class="col-md-3">
                                    <label for="message_type" class="form-label">Type</label>
                                    <select class="form-select" id="message_type" name="message_type" required>
                                        <option value="received">Received</option>
                                        <option value="sent">Sent</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="message_text" class="form-label">Message Text</label>
                                <textarea class="form-control" id="message_text" name="message_text" rows="3" required></textarea>
                            </div>
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="message_read" name="message_read">
                                <label class="form-check-label" for="message_read">Mark as read (only for sent messages)</label>
                            </div>
                            
                            <div class="d-flex justify-content-between">
                                <button type="button" class="btn btn-secondary" onclick="resetMessageForm()">Clear</button>
                                <button type="submit" name="save_message" class="btn btn-primary">Add Message</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Conversations List -->
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0">Existing Conversations</h5>
                    </div>
                    <div class="card-body">
                        <?php foreach ($messages_data['messages'] as $contact_id => $contact): ?>
                            <div class="message-container mb-4">
                                <h5 class="contact-header">
                                    <?php echo htmlspecialchars($contact['name']); ?>
                                </h5>
                                
                                <?php foreach ($contact['conversations'] as $date_index => $conversation): ?>
                                    <div class="message-group mb-3">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div class="message-date">
                                                <?php echo $conversation['date']; ?>
                                            </div>
                                            <a href="?tab=messages&delete_message_group=<?php echo $contact_id . ':' . $date_index; ?>" class="text-danger" onclick="return confirm('Are you sure you want to delete all messages in this conversation?')">
                                                <i class="bi bi-trash"></i> Exclude conversation
                                            </a>
                                        </div>
                                        
                                        <?php foreach ($conversation['msgs'] as $msg_index => $msg): ?>
                                            <div class="message-item message-<?php echo $msg['type']; ?>">
                                                <?php echo htmlspecialchars($msg['text']); ?>
                                                <div class="message-time">
                                                    <?php echo $msg['time']; ?>
                                                    <?php if (isset($msg['read']) && $msg['read']): ?>
                                                        <span class="ms-2">✓ Read</span>
                                                    <?php endif; ?>
                                                </div>
                                                <a href="?tab=messages&delete_message=<?php echo $contact_id . ':' . $date_index . ':' . $msg_index; ?>" class="delete-message" onclick="return confirm('Are you sure you want to delete this message?')">
                                                    <i class="bi bi-x-circle-fill"></i>
                                                </a>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endforeach; ?>
                        
                        <?php if (empty($messages_data['messages'])): ?>
                            <div class="alert alert-info">
                                No conversations found. Add some messages!
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Functions for Notes
        function editNote(note) {
            document.getElementById('note_id').value = note.id;
            document.getElementById('note_title').value = note.title;
            document.getElementById('note_date').value = note.date;
            document.getElementById('note_content').value = note.content;
            
            // Scroll to the form
            document.querySelector('.card-header').scrollIntoView({ behavior: 'smooth' });
        }
        
        function resetNoteForm() {
            document.getElementById('note_id').value = '0';
            document.getElementById('note_title').value = '';
            document.getElementById('note_date').value = '<?php echo date('d/m/Y'); ?>';
            document.getElementById('note_content').value = '';
        }
        
        // Functions for Gallery
        function editPhoto(photo) {
            document.getElementById('photo_id').value = photo.id;
            document.getElementById('photo_title').value = photo.title;
            document.getElementById('photo_date').value = photo.date;
            document.getElementById('photo_description').value = photo.description;
            
            // Show current image preview
            const previewContainer = document.getElementById('photo_preview_container');
            const previewImage = document.getElementById('photo_preview');
            
            previewImage.src = photo.image;
            previewContainer.classList.remove('d-none');
            
            // Scroll to the form
            document.querySelector('.card-header').scrollIntoView({ behavior: 'smooth' });
        }
        
        function resetPhotoForm() {
            document.getElementById('photo_id').value = '0';
            document.getElementById('photo_title').value = '';
            document.getElementById('photo_date').value = '<?php echo date('d/m/Y'); ?>';
            document.getElementById('photo_description').value = '';
            document.getElementById('photo_image').value = '';
            
            // Hide image preview
            document.getElementById('photo_preview_container').classList.add('d-none');
        }
        
        // Functions for Calls
        function editCall(call) {
            document.getElementById('call_id').value = call.id;
            document.getElementById('call_name').value = call.name;
            document.getElementById('call_number').value = call.number;
            document.getElementById('call_date').value = call.date;
            document.getElementById('call_time').value = call.time;
            document.getElementById('call_duration').value = call.duration;
            document.getElementById('call_type').value = call.type;
            document.getElementById('call_description').value = call.description;
            
            // Show current audio preview
            const previewContainer = document.getElementById('call_audio_preview_container');
            const previewAudio = document.getElementById('call_audio_preview');
            
            previewAudio.src = call.audio;
            previewContainer.classList.remove('d-none');
            
            // Scroll to the form
            document.querySelector('.card-header').scrollIntoView({ behavior: 'smooth' });
        }
        
        function resetCallForm() {
            document.getElementById('call_id').value = '0';
            document.getElementById('call_name').value = '';
            document.getElementById('call_number').value = '';
            document.getElementById('call_date').value = '<?php echo date('d/m/Y'); ?>';
            document.getElementById('call_time').value = '<?php echo date('H:i'); ?>';
            document.getElementById('call_duration').value = '0:00';
            document.getElementById('call_type').value = 'incoming';
            document.getElementById('call_description').value = '';
            document.getElementById('call_audio').value = '';
            
            // Hide audio preview
            document.getElementById('call_audio_preview_container').classList.add('d-none');
        }
        
        // Functions for Messages
        function resetMessageForm() {
            document.getElementById('contact_id').value = 'unknown';
            document.getElementById('message_date').value = '<?php echo date('F d, Y'); ?>';
            document.getElementById('message_time').value = '<?php echo date('H:i'); ?>';
            document.getElementById('message_type').value = 'received';
            document.getElementById('message_text').value = '';
            document.getElementById('message_read').checked = false;
            
            // Hide new contact field
            document.querySelectorAll('.new-contact-field').forEach(field => {
                field.classList.add('d-none');
            });
        }
        
        // Event to show/hide new contact field
        document.addEventListener('DOMContentLoaded', function() {
            const contactSelect = document.getElementById('contact_id');
            
            if (contactSelect) {
                contactSelect.addEventListener('change', function() {
                    const newContactFields = document.querySelectorAll('.new-contact-field');
                    
                    if (this.value === 'new') {
                        newContactFields.forEach(field => {
                            field.classList.remove('d-none');
                        });
                        document.getElementById('contact_name').setAttribute('required', 'required');
                    } else {
                        newContactFields.forEach(field => {
                            field.classList.add('d-none');
                        });
                        document.getElementById('contact_name').removeAttribute('required');
                    }
                });
            }
        });
    </script>
</body>
</html>